// Copyright 2024 Buf Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"go/format"
	"io"
	"net/http"
	"os"
	"sort"
	"strconv"
	"strings"
)

const (
	programName        = "gen-spdx-go"
	licenseListVersion = "v3.23"
)

var (
	httpClient = &http.Client{}

	licenseListURL = "https://raw.githubusercontent.com/spdx/license-list-data/" +
		licenseListVersion +
		"/json/licenses.json"
)

func main() {
	if err := run(context.Background()); err != nil {
		if errString := err.Error(); errString != "" {
			_, _ = fmt.Fprintln(os.Stderr, errString)
		}
		os.Exit(1)
	}
}

func run(ctx context.Context) error {
	licenses, err := getLicenses(ctx)
	if err != nil {
		return err
	}
	data, err := getGolangFileData(licenses)
	if err != nil {
		return err
	}
	_, err = os.Stdout.Write(data)
	return err
}

func getGolangFileData(
	licenses []license,
) ([]byte, error) {
	buffer := bytes.NewBuffer(nil)
	_, _ = buffer.WriteString(`// Code generated by ` + programName + `. DO NOT EDIT.

package spdx

// LicenseListVersion is the version of the SPDX license list.
//
// See https://spdx.org/licenses.
const LicenseListVersion = `)
	_, _ = buffer.WriteString(strconv.Quote(licenseListVersion))
	_, _ = buffer.WriteString(`

// lowercaseIDToLicense contains a map from lowercased ID to License.
//
// Licenses will be unique by License ID in a case-insensitive manner.
var lowercaseIDToLicense = map[string]License{
`)
	for _, license := range licenses {
		writeLicense(buffer, license)
	}
	_, _ = buffer.WriteString("}")
	return format.Source(buffer.Bytes())
}

func writeLicense(buffer *bytes.Buffer, license license) {
	_, _ = buffer.WriteString(strconv.Quote(strings.ToLower(license.ID)) + `: {` + "\n")
	_, _ = buffer.WriteString("ID: " + strconv.Quote(license.ID) + ", " + "\n")
	_, _ = buffer.WriteString("Name: " + strconv.Quote(license.Name) + ", " + "\n")
	_, _ = buffer.WriteString("Reference: " + strconv.Quote(license.Reference) + ", " + "\n")
	_, _ = buffer.WriteString("ReferenceNumber: " + strconv.Itoa(license.ReferenceNumber) + ", " + "\n")
	_, _ = buffer.WriteString("DetailsURL: " + strconv.Quote(license.DetailsURL) + ", " + "\n")
	_, _ = buffer.WriteString("Deprecated: " + strconv.FormatBool(license.Deprecated) + ", " + "\n")
	_, _ = buffer.WriteString("SeeAlso: []string{" + "\n")
	for _, seeAlso := range license.SeeAlso {
		_, _ = buffer.WriteString(strconv.Quote(seeAlso) + ", " + "\n")
	}
	_, _ = buffer.WriteString("},\n")
	_, _ = buffer.WriteString("OSIApproved: " + strconv.FormatBool(license.OSIApproved) + ", " + "\n")
	_, _ = buffer.WriteString("},\n")
}

func getLicenses(ctx context.Context) (_ []license, retErr error) {
	request, err := http.NewRequestWithContext(ctx, http.MethodGet, licenseListURL, nil)
	if err != nil {
		return nil, err
	}
	response, err := httpClient.Do(request)
	if err != nil {
		return nil, err
	}
	defer func() {
		retErr = errors.Join(retErr, response.Body.Close())
	}()
	if response.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("expected HTTP status code %d to be %d", response.StatusCode, http.StatusOK)
	}
	data, err := io.ReadAll(response.Body)
	if err != nil {
		return nil, err
	}
	licenseList := &licenseList{}
	if err := json.Unmarshal(data, licenseList); err != nil {
		return nil, err
	}
	lowercaseIDMap := make(map[string]struct{})
	for _, license := range licenseList.Licenses {
		lowercaseID := strings.ToLower(license.ID)
		if _, ok := lowercaseIDMap[lowercaseID]; ok {
			return nil, fmt.Errorf("duplicate lowercase ID: %q", lowercaseID)
		}
		lowercaseIDMap[lowercaseID] = struct{}{}
	}
	sort.Slice(
		licenseList.Licenses,
		func(i int, j int) bool {
			return licenseList.Licenses[i].ID < licenseList.Licenses[j].ID
		},
	)
	return licenseList.Licenses, nil
}

type licenseList struct {
	Licenses []license `json:"licenses,omitempty" yaml:"licenses,omitempty"`
}

// Copied so we have no dependencies on spdx package.
type license struct {
	ID              string   `json:"licenseId,omitempty"`
	Name            string   `json:"name,omitempty"`
	Reference       string   `json:"reference,omitempty"`
	ReferenceNumber int      `json:"referenceNumber,omitempty"`
	DetailsURL      string   `json:"detailsUrl,omitempty"`
	Deprecated      bool     `json:"isDeprecatedLicenceId,omitempty"`
	SeeAlso         []string `json:"seeAlso,omitempty"`
	OSIApproved     bool     `json:"isOsiApproved,omitempty"`
}
