/*
 * FormTemplate.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1998-2002 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, rinaldo, suhler.
 *
 * Version:  2.2
 * Created by suhler on 98/10/26
 * Last modified by suhler on 02/12/19 11:36:50
 */

package sunlabs.brazil.template;

import sunlabs.brazil.server.Server;
import java.io.Serializable;

/**
 * Template class for substituting default values into html forms.
 * This class is used by the TemplateHandler.
 * 
 * The default values in form elements are replaced by the request property that
 * matches the field name. The following field elements are processed:
 * <ul>
 * <li> &lt;input name=<i>x</i> value=<i>y</i>&gt;
 * <li> &lt;input type=input name=<i>x</i> value=<i>y</i>&gt;
 * <li> &lt;input type=radio name=<i>x</i> value=<i>y</i>&gt;
 * <li> &lt;option value=<i>x</i>&gt;
 * </ul>
 * In all cases, the <code>value</code> attribute must be present.
 * additional information is provided below.
 * 
 * @author		Stephen Uhler
 * @version		@(#)FormTemplate.java	2.2
 */
public class FormTemplate extends Template implements Serializable {
    String selectName;      // The name of the current select variable
    String selectValue;     // The name of the current select variable value
    int total;              // total for elements examined
    int changed;            // elements whose initial values have been changed

    /**
     * Save a reference to our request properties.
     */

    public boolean init(RewriteContext hr) {
        selectName = null;
        total = 0;
        changed = 0;

        return true;
    }

    /**
     * Look for &lt;input name=[x] value=[v]&gt; and replace the
     * value with the entry in the request properties. If no value is supplied,
     * no substitution is done.
     * If <code>value</code> contains any ${..} constructs, the substituted
     * value is used instead of the value in the corrosponding request property.
     */
    public void tag_input(RewriteContext hr) {
        total++;

	String origValue = hr.get("value");
	if (origValue != null) {
	    hr.put("value", origValue);
	}

        String name = hr.get("name");
        if (name == null) {
            return;
        } else {
	    hr.put("name", name);
	}

        String value = hr.request.props.getProperty(name);
        if (value == null) {
            return;
        }

        changed++;
        String type = hr.get("type", false);
	if ("radio".equals(type) || "checkbox".equals(type)) {
	    log(hr,type + " " + value + " ? " + hr.get("value", false));
	    if (value.equals(origValue)) {
		hr.put("checked", "");
	    } else {
		hr.remove("checked");
	    }
	} else {
	    hr.put("value", value);
	    String size = hr.get("size");
            if (size != null) {
	       hr.put("size",size);
	    }
	}
    }

    /**
     * Remember the variable name for the next group of option tags.
     */
    public void tag_select(RewriteContext hr) {
        selectName = hr.get("name");

        if (selectName != null) {
            selectValue = hr.request.props.getProperty(selectName);
	    hr.put("name", selectName);

            log(hr, 
                "For selection (" + selectName + ") have value :" 
                + selectValue);
        }
    }

    /**
     * Forget the variable name for the next group of option tags
     */
    public void tag_slash_select(RewriteContext hr) {
        selectName = null;
    }

    /**
     * Look at the option tag, set the "selected" attribute as needed.
     * In order for this to work, the VALUE tag *must* be used
     * Do ${...} substitutions on the value.
     */
    public void tag_option(RewriteContext hr) {
        String value = hr.get("value");

        if (value != null) {
	    hr.put("value", value);
        }

        if (selectName == null || selectValue == null || value == null) {
            return;
        }
        if (value.equals(selectValue)) {
            hr.put("selected", "");
        } else {
            hr.remove("selected");
        }
    }

    /**
     * This is for debugging only !!
     */

    public boolean done(RewriteContext hr) {
        log(hr, hr.request.url + " (form template) " + changed + "/" + total + " changed");

        return true;
    }

    /**
     * simple interface to server logging
     */

    private void log(RewriteContext hr, String msg) {
        hr.request.log(Server.LOG_DIAGNOSTIC, 
                       hr.prefix + "formTemplate: " + msg);
    }
}
