(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
module Microsoft.FSharp.Compiler.Pickle 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Il = Microsoft.Research.AbstractIL.IL 
module Ilx    = Microsoft.Research.AbstractIL.Extensions.ILX.Types
module Ccuthunk = Microsoft.FSharp.Compiler.CcuThunk 
F#*) 

open Tast
open Tastops
open Lib
open Ildiag
open Range
open Ccuthunk

let verbose = false

let pickle_failwith str = failwith ("Error reading/writing metadata for an F# compiled DLL. Was the DLL compiled with an earlier version of the F# compiler? (" ^ str ^ ")")

(*---------------------------------------------------------------------------
 * Basic pickle/unpickle state
 *------------------------------------------------------------------------- *)

type 'a tbl = 
    { name: string;
      tbl: ('a, int) Hashtbl.t;
      mutable rows: 'a list;
      mutable count: int }

(* inline this to get known-type-information through to the Hashtbl.create *)
let (*F# inline F#*) new_tbl n = 
  { name = n;
    tbl = (Hashtbl.create 20);
    rows=[];
    count=0; }

let get_tbl tbl = Array.of_list (List.rev tbl.rows)
let tbl_size tbl = List.length tbl.rows

let add_entry tbl x =
  let n = tbl.count in 
  tbl.count <- tbl.count + 1;
  Hashtbl.add tbl.tbl x n;
  tbl.rows <- x :: tbl.rows;
  n

let find_or_add_entry tbl x =
  if Hashtbl.mem tbl.tbl x then Hashtbl.find tbl.tbl x 
  else add_entry tbl x


let tbl_find tbl x =
  if Hashtbl.mem tbl.tbl x then Hashtbl.find tbl.tbl x 
  else raise Not_found

let tbl_mem tbl x = Hashtbl.mem tbl.tbl x 

type 'a itbl = 
    { itbl_name: string;
      itbl_rows: 'a array }

let new_itbl n r = { itbl_name=n; itbl_rows=r }

type 'a osgn_outmap = ObservableNodeOutMap of ('a osgn -> int) * ('a osgn -> string) * string * int tbl 
type 'a osgn_inmap = ObservableNodeInMap of string * 'a osgn array 
let new_osgn_inmap nm n = ObservableNodeInMap (nm, Array.init n (fun i -> new_unlinked_osgn() ))
(* inline this to get known-type-information through to the Hashtbl.create *)
let (*F# inline F#*) new_osgn_outmap f g nm = ObservableNodeOutMap (f, g,nm, new_tbl nm)
let osgn_outmap_size (ObservableNodeOutMap(_,_,_,x)) = tbl_size x

type outstate = 
  { os: Bytes.Bytebuf.t; 
    oscope: ccu;
    occus: string tbl; 
    otycons: tycon_spec_data osgn_outmap; 
    otypars: typar_spec_data osgn_outmap; 
    ovals: val_spec_data osgn_outmap;
    ostrings: string tbl; 
    opubpaths: (int list * int) tbl; 
    onlpaths: (int * int list) tbl; 
    osimpletyps: (int * int) tbl;
  }

type instate = 
  { is: Bytes.Bytestream.t; 
    iilscope: Il.scope_ref;
    iccus: ccu itbl; 
    itycons: tycon_spec_data osgn_inmap;  
    itypars: typar_spec_data osgn_inmap; 
    ivals: val_spec_data osgn_inmap;
    istrings: string itbl;
    ipubpaths: public_path itbl; 
    inlpaths: nonlocal_path itbl; 
    isimpletyps: Tast.typ itbl;
  }

(*---------------------------------------------------------------------------
 * Basic pickle/unpickle operations
 *------------------------------------------------------------------------- *)
 
let p_byte b st = Bytes.Bytebuf.emit_int_as_byte st.os b
let u_byte st = Bytes.Bytestream.read_byte st.is

type 'a pickler = 'a -> outstate -> unit
type 'a unpickler = instate -> 'a

let p_bool b st = p_byte (if b then 1 else 0) st
let u_bool st = let b = u_byte st in (b = 1) 

let p_void (os: outstate) = ()
let u_void (is: instate) = ()

let p_unit () (os: outstate) = ()
let u_unit (is: instate) = ()

let prim_p_int32 i st = 
  p_byte (b0 i) st;
  p_byte (b1 i) st;
  p_byte (b2 i) st;
  p_byte (b3 i) st

let prim_u_int32 st = 
  let b0 =  (u_byte st) in
  let b1 =  (u_byte st) in
  let b2 =  (u_byte st) in
  let b3 =  (u_byte st) in
  !!!b0 ||| (!!!b1 <<< 8) ||| (!!!b2 <<< 16) ||| (!!!b3 <<< 24)

(* compress integers according to the same scheme used by CLR metadata *)
(* This halves the size of pickled data *)
let p_int32 n st = 
  if n >= 0l &  n <= 0x7Fl then 
    p_byte (b0 n) st
  else if n >= 0x80l & n <= 0x3FFFl then  begin
    p_byte (Int32.to_int (0x80l ||| (n lsr 8))) st; 
    p_byte (Int32.to_int (n &&& 0xFFl)) st 
  end else begin
    p_byte 0xFF st;
    prim_p_int32 n st
  end

let u_int32 st = 
  let b0 = u_byte st in 
  if b0 <= 0x7F then !!!b0 
  else if b0 <= 0xbf then 
    let b0 = !!!b0 &&& !!!0x7f in 
    let b1 = !!!(u_byte st) in 
    (b0 <<< 8) ||| b1
  else  begin
    assert(b0 = 0xFF);
    prim_u_int32 st
  end

let p_bytes s st = 
  let len = Bytes.length s in
  p_int32 (Int32.of_int len) st;
  Bytes.Bytebuf.emit_bytes st.os s

let u_bytes st = 
  let n = Int32.to_int (u_int32 st) in 
  Bytes.Bytestream.read_bytes st.is n

let pprim_string s st = 
    let bytes = Bytes.string_as_utf8_bytes s in
    let len = Bytes.length bytes in
    p_int32 (Int32.of_int len) st;
    Bytes.Bytebuf.emit_bytes st.os bytes
let uprim_string st = 
    let len = Int32.to_int (u_int32 st) in 
    Bytes.Bytestream.read_utf8_bytes_as_string st.is len

let p_int c st = p_int32 (Int32.of_int c) st
let u_int st = Int32.to_int (u_int32 st)

let p_int8 i st = p_int32 (Nums.i8_to_i32 i) st
let u_int8 st = Nums.i32_to_i8 (u_int32 st)

let p_uint8 i st = p_byte (Nums.u8_to_int i) st
let u_uint8 st = Nums.int_to_u8 (u_byte st)

let p_int16 i st = p_int32 (Nums.i16_to_i32 i) st
let u_int16 st = Nums.i32_to_i16 (u_int32 st)

let p_uint16 x st = p_int32 (Nums.u16_to_i32 x) st
let u_uint16 st = Nums.i32_to_u16 (u_int32 st)

let p_uint32 x st = p_int32 (Nums.u32_to_i32 x) st
let u_uint32 st = Nums.i32_to_u32 (u_int32 st)

let p_int64 i st = 
  p_int32 (Int64.to_int32 (Int64.logand i 0xFFFFFFFFL)) st;
  p_int32 (Int64.to_int32 (Int64.shift_right_logical i 32)) st

let u_int64 st = 
  let b1 = Int64.logand (Int64.of_int32 (u_int32 st)) 0xFFFFFFFFL in
  let b2 = Int64.of_int32 (u_int32 st) in
  Int64.logor b1 (Int64.shift_left b2 32)

let p_uint64 x st = p_int64 (Nums.u64_to_i64 x) st
let u_uint64 st = Nums.i64_to_u64 (u_int64 st)

let p_single i st = p_int32 (Nums.ieee32_to_bits i) st
let u_single st = Nums.bits_to_ieee32 (u_int32 st)

let p_float64 i st = p_int64 (Int64.bits_of_float i) st
let u_float64 st = Int64.float_of_bits (u_int64 st)

let p_ieee64 i st = p_int64 (Nums.ieee64_to_bits i) st
let u_ieee64 st = Nums.bits_to_ieee64 (u_int64 st)

let p_char i st = p_uint16 (Nums.unichar_to_u16 i) st
let u_char st = Nums.u16_to_unichar (u_uint16 st)

let (*F# inline F#*) p_tup2 p1 p2 (a,b) (st:outstate) = (p1 a st : unit); (p2 b st : unit)
let (*F# inline F#*) p_tup3 p1 p2 p3 (a,b,c) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit)
let (*F# inline F#*)  p_tup4 p1 p2 p3 p4 (a,b,c,d) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit)
let (*F# inline F#*)  p_tup5 p1 p2 p3 p4 p5 (a,b,c,d,e) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit)
let (*F# inline F#*)  ptup6 p1 p2 p3 p4 p5 p6 (a,b,c,d,e,f) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit)
let (*F# inline F#*)  ptup7 p1 p2 p3 p4 p5 p6 p7 (a,b,c,d,e,f,x7) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit)
let (*F# inline F#*)  ptup8 p1 p2 p3 p4 p5 p6 p7 p8 (a,b,c,d,e,f,x7,x8) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit); (p8 x8 st : unit)
let (*F# inline F#*)  ptup9 p1 p2 p3 p4 p5 p6 p7 p8 p9 (a,b,c,d,e,f,x7,x8,x9) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit); (p8 x8 st : unit); (p9 x9 st : unit)
let (*F# inline F#*)  ptup10 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 (a,b,c,d,e,f,x7,x8,x9,x10) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit); (p8 x8 st : unit); (p9 x9 st : unit); (p10 x10 st : unit)
let (*F# inline F#*)  ptup11 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 (a,b,c,d,e,f,x7,x8,x9,x10,x11) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit); (p8 x8 st : unit); (p9 x9 st : unit); (p10 x10 st : unit); (p11 x11 st : unit)
let (*F# inline F#*)  ptup12 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 (a,b,c,d,e,f,x7,x8,x9,x10,x11,x12) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit); (p8 x8 st : unit); (p9 x9 st : unit); (p10 x10 st : unit); (p11 x11 st : unit); (p12 x12 st : unit)
let (*F# inline F#*)  ptup13 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 p13 (a,b,c,d,e,f,x7,x8,x9,x10,x11,x12,x13) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit); (p8 x8 st : unit); (p9 x9 st : unit); (p10 x10 st : unit); (p11 x11 st : unit); (p12 x12 st : unit); (p13 x13 st : unit)
let (*F# inline F#*)  ptup14 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 p13 p14 (a,b,c,d,e,f,x7,x8,x9,x10,x11,x12,x13,x14) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit); (p8 x8 st : unit); (p9 x9 st : unit); (p10 x10 st : unit); (p11 x11 st : unit); (p12 x12 st : unit); (p13 x13 st : unit) ; (p14 x14 st : unit)
let (*F# inline F#*)  ptup15 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 p13 p14 p15 (a,b,c,d,e,f,x7,x8,x9,x10,x11,x12,x13,x14,x15) (st:outstate) = (p1 a st : unit); (p2 b st : unit); (p3 c st : unit); (p4 d st : unit); (p5 e st : unit); (p6 f st : unit); (p7 x7 st : unit); (p8 x8 st : unit); (p9 x9 st : unit); (p10 x10 st : unit); (p11 x11 st : unit); (p12 x12 st : unit); (p13 x13 st : unit) ; (p14 x14 st : unit); (p15 x15 st : unit)

let (*F# inline F#*)  u_tup2 p1 p2 (st:instate) = let a = p1 st in let b = p2 st in (a,b)
let (*F# inline F#*)  u_tup3 p1 p2 p3 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in (a,b,c)
let (*F# inline F#*) u_tup4 p1 p2 p3 p4 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in (a,b,c,d)
let (*F# inline F#*) u_tup5 p1 p2 p3 p4 p5 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in (a,b,c,d,e)
let (*F# inline F#*) utup6 p1 p2 p3 p4 p5 p6 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in (a,b,c,d,e,f)
 let (*F# inline F#*) utup7 p1 p2 p3 p4 p5 p6 p7 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in (a,b,c,d,e,f,x7)
 let (*F# inline F#*) utup8 p1 p2 p3 p4 p5 p6 p7 p8 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in let x8 = p8 st in 
  (a,b,c,d,e,f,x7,x8)
 let (*F# inline F#*) utup9 p1 p2 p3 p4 p5 p6 p7 p8 p9 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in let x8 = p8 st in 
  let x9 = p9 st in (a,b,c,d,e,f,x7,x8,x9)
let (*F# inline F#*) utup10 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in let x8 = p8 st in 
  let x9 = p9 st in let x10 = p10 st in (a,b,c,d,e,f,x7,x8,x9,x10)
let (*F# inline F#*) utup11 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in let x8 = p8 st in 
  let x9 = p9 st in let x10 = p10 st in let x11 = p11 st in (a,b,c,d,e,f,x7,x8,x9,x10,x11)
let (*F# inline F#*) utup12 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in let x8 = p8 st in 
  let x9 = p9 st in let x10 = p10 st in let x11 = p11 st in let x12 = p12 st in 
  (a,b,c,d,e,f,x7,x8,x9,x10,x11,x12)
let (*F# inline F#*) utup13 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 p13 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in let x8 = p8 st in 
  let x9 = p9 st in let x10 = p10 st in let x11 = p11 st in let x12 = p12 st in let x13 = p13 st in 
  (a,b,c,d,e,f,x7,x8,x9,x10,x11,x12,x13)
let (*F# inline F#*) utup14 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 p13 p14 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in let x8 = p8 st in 
  let x9 = p9 st in let x10 = p10 st in let x11 = p11 st in let x12 = p12 st in let x13 = p13 st in 
  let x14 = p14 st in 
  (a,b,c,d,e,f,x7,x8,x9,x10,x11,x12,x13,x14)
let (*F# inline F#*) utup15 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 p13 p14 p15 (st:instate) =
  let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in
  let e = p5 st in let f = p6 st in let x7 = p7 st in let x8 = p8 st in 
  let x9 = p9 st in let x10 = p10 st in let x11 = p11 st in let x12 = p12 st in let x13 = p13 st in 
  let x14 = p14 st in let x15 = p15 st in 
  (a,b,c,d,e,f,x7,x8,x9,x10,x11,x12,x13,x14,x15)

(*---------------------------------------------------------------------------
 * Pickle/unpickle operations for observably shared graph nodes
 *------------------------------------------------------------------------- *)

(* exception Nope *)

(* ctxt is for debugging *)
let posgn_ref (ctxt:string) (ObservableNodeOutMap(stampf,namef,nm,keyTable)) x st = 
    let idx = find_or_add_entry keyTable (stampf x) in 
    if ((idx = 38 or idx = 3) && nm = "otycons") then (dprintf5 "idx %d#%d in table %s has name '%s' and is referenced from context %s\n" idx (stampf x) nm (namef x) ctxt; (* raise Nope *) ); 
    p_int idx st

let uosgn_ref (ObservableNodeInMap(nm,arr)) st = 
    let n = u_int st in 
    if n < 0 or n >= Array.length arr then pickle_failwith ("uosgn_ref: out of range, table = "^nm^", n = "^string_of_int n); 
    arr.(n)

let posgn_decl (ObservableNodeOutMap(stampf,namef,nm,keyTable)) p x st = 
    let stamp = stampf x in 
    let idx = find_or_add_entry keyTable stamp in
    (* dprintf4 "decl %d#%d in table %s has name %s\n" idx (stampf x) nm (namef x);  *)
    p_tup2 p_int p (idx,deref_osgn x) st

let uosgn_decl (ObservableNodeInMap(nm,arr)) u st = 
    let idx,data = u_tup2 u_int u st in 
  (*   dprintf2 "unpickling osgn %d in table %s\n" idx nm; *)
    let res = arr.(idx) in 
    link_osgn res data;
    res

(*---------------------------------------------------------------------------
 * Pickle/unpickle operations for interned nodes in the term DAG
 *------------------------------------------------------------------------- *)

let encode_uniq tbl key = find_or_add_entry tbl key
let lookup_uniq tbl n = 
  let arr = tbl.itbl_rows in
  if n < 0 or n >= Array.length arr then pickle_failwith ("lookup_uniq in table "^tbl.itbl_name^" out of range, n = "^string_of_int n^ ", sizeof(tab) = " ^ string_of_int (Array.length arr)); 
  arr.(n)

(*---------------------------------------------------------------------------
 * Pickle/unpickle lists
 *------------------------------------------------------------------------- *)
 
let (>>) f g x = g (f x)

(* nb. must be tail-recursive *)
let rec p_list f x st =
  match x with 
  | [] -> p_byte 0 st
  | h :: t -> p_byte 1 st; f h st; p_list f t st
      
(* nb. must be tail-recursive *)
let rec ulist_aux f acc st = 
  let tag = u_byte st in match tag with
  | 0 -> List.rev acc
  | 1 -> let a = f st in ulist_aux f (a::acc) st 
  | n -> pickle_failwith ("u_list: found number " ^ string_of_int n)

let u_list f st = ulist_aux f [] st
 
(* Mark up default constraints with a priority in reverse order: last gets 0 etc. See comment on TTyparDefaultsToType *)
let rec ulisti_aux f acc st = 
  let tag = u_byte st in match tag with
  | 0 -> List.rev (list_mapi (fun i x -> x i) acc)
  | 1 -> let a = f st in ulisti_aux f (a::acc) st 
  | n -> pickle_failwith ("ulisti: found number " ^ string_of_int n)

let ulisti f st = ulisti_aux f [] st
 
(*---------------------------------------------------------------------------
 * Pickle/unpickle operations for standard data structure builders
 *------------------------------------------------------------------------- *)

let p_wrap (f: 'a -> 'b) (p : 'b pickler) : 'a pickler = (fun x st -> p (f x) st)
let u_wrap (f: 'b -> 'a) (u : 'b unpickler) : 'a unpickler = (fun st -> f (u st))

let p_array f = p_wrap Array.to_list (p_list f)
let u_array f = u_wrap Array.of_list (u_list f)

let p_option f x st =
  match x with 
  | None -> p_byte 0 st
  | Some h -> p_byte 1 st; f h st

let u_option f st = 
  let tag = u_byte st in match tag with
  | 0 -> None
  | 1 -> Some (f st)
  | n -> pickle_failwith ("u_option: found number " ^ string_of_int n)

let plazy_immediate p = p_wrap Lazy.force p
let ulazy_immediate u = u_wrap Lazy.lazy_from_val u

let p_hole () = 
  let h = ref (None : 'a pickler option) in
  (fun f -> h := Some f),(fun x st -> match !h with Some f -> f x st | None -> pickle_failwith "p_hole: unfilled hole")

let u_hole () = 
  let h = ref (None : 'a unpickler option) in
  (fun f -> h := Some f),(fun st -> match !h with Some f -> f st | None -> pickle_failwith "u_hole: unfilled hole")

(*---------------------------------------------------------------------------
 * Pickle/unpickle F# interface data 
 *------------------------------------------------------------------------- *)

(* Strings *)
(* A huge number of these occur in pickled F# data, so make them unique *)
let encode_string stringTab x = encode_uniq stringTab x
let decode_string x = x
let lookup_string stringTab x = lookup_uniq stringTab x
let pencoded_string = pprim_string
let uencoded_string = uprim_string
let p_string s st = p_int (encode_string st.ostrings s) st
let u_string st   = lookup_uniq st.istrings (u_int st)

(* CCU References *)
(* A huge number of these occur in pickled F# data, so make them unique *)
let encode_ccuref ccuTab x = encode_uniq ccuTab (name_of_ccu x) 
let decode_ccuref x = x
let lookup_ccuref ccuTab x = lookup_uniq ccuTab x
let pencoded_ccuref = pprim_string
let uencoded_ccuref = uprim_string
let pccuref s st = p_int (encode_ccuref st.occus s) st
let uccuref st   = lookup_uniq st.iccus (u_int st)

(* References to public items in this module *)
(* A huge number of these occur in pickled F# data, so make them unique *)
let encode_pubpath stringTab pubpathTab (PubPath(a,b)) = encode_uniq pubpathTab (List.map (encode_string stringTab) a, encode_string stringTab b)
let decode_pubpath stringTab (a,b) = PubPath(List.map (lookup_string stringTab) a, lookup_string stringTab b)
let lookup_pubpath pubpathTab x = lookup_uniq pubpathTab x
let pencoded_pubpath = p_tup2 (p_list p_int) p_int
let uencoded_pubpath = u_tup2 (u_list u_int) u_int
let ppubpath x st = p_int (encode_pubpath st.ostrings st.opubpaths x) st
let upubpath st = lookup_uniq st.ipubpaths (u_int st)

(* References to other modules *)
(* A huge number of these occur in pickled F# data, so make them unique *)
let encode_nlpath ccuTab stringTab nlpathTab (NLPath(a,b)) = encode_uniq nlpathTab (encode_ccuref ccuTab a, List.map (encode_string stringTab) b)
let decode_nlpath ccuTab stringTab (a,b) = NLPath(lookup_ccuref ccuTab a, List.map (lookup_string stringTab) b)
let lookup_nlpath nlpathTab x = lookup_uniq nlpathTab x
let pencoded_nlpath = p_tup2 p_int (p_list p_int)
let uencoded_nlpath = u_tup2 u_int (u_list u_int)
let pnlpath x st = p_int (encode_nlpath st.occus st.ostrings st.onlpaths x) st
let unlpath st = lookup_uniq st.inlpaths (u_int st)

(* Simple types are types like "int", represented as TType(Ref_nonlocal(...,"int"),[]). *)
(* A huge number of these occur in pickled F# data, so make them unique. *)
let encode_simpletyp ccuTab stringTab nlpathTab simpletypTab (a,b) = encode_uniq simpletypTab (encode_nlpath ccuTab stringTab nlpathTab a, encode_string stringTab b)
let decode_simpletyp ccuTab stringTab nlpathTab (a,b) = TType_app(mk_nonlocal_ref (lookup_nlpath nlpathTab a) (lookup_string stringTab b),[])
let lookup_simpletyp simpletypTab x = lookup_uniq simpletypTab x
let pencoded_simpletyp = p_tup2 p_int p_int
let uencoded_simpletyp = u_tup2 u_int u_int
let psimpletyp x st = p_int (encode_simpletyp st.occus st.ostrings st.onlpaths st.osimpletyps x) st
let usimpletyp st = lookup_uniq st.isimpletyps (u_int st)

type sizes = int * int * int 
let pickle_obj_with_dangling_ccus scope p x =
  let ccuNameTab,(sizes: sizes),stringTab,pubpathTab,nlpathTab,simpletypTab,phase1bytes =
    let st1 = 
      { os = Bytes.Bytebuf.create 100000; 
        oscope=scope;
        occus= new_tbl "occus"; 
        otycons=new_osgn_outmap stamp_of_tycon name_of_tycon "otycons"; 
        otypars=new_osgn_outmap stamp_of_typar (fun tp -> bufs (fun buf -> Printf.bprintf buf "%s - %a" (Tastops.display_name_of_typar tp) boutput_range (range_of_typar tp))) "otypars"; 
        ovals=new_osgn_outmap stamp_of_val (fun v -> bufs (fun buf -> Printf.bprintf buf "%s - %a" (name_of_val v) boutput_range (range_of_val v))) "ovals"; 
        ostrings=new_tbl "ostrings";
        onlpaths=new_tbl "onlpaths";  
        opubpaths=new_tbl "opubpaths";  
        osimpletyps=new_tbl "osimpletyps";  
        (* REINSTATE: odecomps=new_osgn_outmap stamp_of_decomp name_of_decomp "odecomps"; *) } in
    p x st1;
    let sizes = 
      osgn_outmap_size st1.otycons,
      osgn_outmap_size st1.otypars,
      osgn_outmap_size st1.ovals  in 
    st1.occus, sizes, st1.ostrings, st1.opubpaths,st1.onlpaths, st1.osimpletyps, Bytes.Bytebuf.close st1.os in
  let phase2data = (get_tbl ccuNameTab,sizes,get_tbl stringTab,get_tbl pubpathTab,get_tbl nlpathTab,get_tbl simpletypTab,phase1bytes) in 
  let phase2bytes = 
    let st2 = 
     { os = Bytes.Bytebuf.create 100000; 
       oscope=scope;
       occus= new_tbl "occus (fake)"; 
       otycons=new_osgn_outmap stamp_of_tycon name_of_tycon "otycons"; 
       otypars=new_osgn_outmap stamp_of_typar Tastops.display_name_of_typar "otypars"; 
       ovals=new_osgn_outmap stamp_of_val name_of_val "ovals";
       ostrings=new_tbl "ostrings (fake)";
       opubpaths=new_tbl "opubpaths (fake)";
       onlpaths=new_tbl "onlpaths (fake)";
       osimpletyps=new_tbl "osimpletyps (fake)"; } in
    ptup7
      (p_array pencoded_ccuref) 
      (p_tup3 p_int p_int p_int) 
      (p_array pencoded_string) 
      (p_array pencoded_pubpath) 
      (p_array pencoded_nlpath) 
      (p_array pencoded_simpletyp) 
      p_bytes 
      phase2data st2;
    Bytes.Bytebuf.close st2.os in 
  phase2bytes
  
let check ilscope (ObservableNodeInMap(nm,arr)) =
  arr |> Array.iteri (fun i n -> 
    if not (osgn_is_linked n) then 
      dprintf4 "*** unpickle: osgn %d in table %s with IL scope %a had no matching declaration (was not fixed up)\nPlease report this warning. (Note for compiler developers: to get information about which item this index relates to, enable the conditional in Pickle.posgn_ref to refer to the given index number and recompile an identical copy of the source for the DLL containing the data being unpickled.  A message will then be printed indicating the name of the item.\n" i nm Ilprint.output_scoref ilscope
  ) 
    
let pstrings = p_list p_string
let ustrings = u_list u_string

let pints = p_list p_int
let uints = u_list u_int

let unpickle_obj_with_dangling_ccus ilscope u phase2bytes =
  let phase2data = 
    let st2 = 
     { is = Bytes.Bytestream.of_bytes phase2bytes 0 (Bytes.length phase2bytes); 
       iilscope= ilscope;
       iccus= new_itbl "iccus (fake)" [| |]; 
       itycons= new_osgn_inmap "itycons" 0; 
       itypars= new_osgn_inmap "itypars" 0; 
       ivals= new_osgn_inmap "ivals" 0;
       (* imoduls= new_osgn_inmap "imoduls" 0; *)
       istrings = new_itbl "istrings (fake)" [| |]; 
       inlpaths = new_itbl "inlpaths (fake)" [| |]; 
       ipubpaths = new_itbl "ipubpaths (fake)" [| |]; 
       isimpletyps = new_itbl "isimpletyps (fake)" [| |]; 
(*      REINSTATE:  idecomps = new_osgn_inmap "idecomps" 0;  *) } in
     utup7
       (u_array uencoded_ccuref) 
       (u_tup3 u_int u_int u_int) 
       (u_array uencoded_string) 
       (u_array uencoded_pubpath) 
       (u_array uencoded_nlpath) 
       (u_array uencoded_simpletyp) 
       u_bytes st2 in
  let ccuNameTab,sizes,stringTab,pubpathTab,nlpathTab,simpletypTab,phase1bytes = phase2data in 
  let ccuTab = new_itbl "iccus" (Array.map new_delayed_ccu_thunk ccuNameTab) in
  let stringTab = new_itbl "istrings" (Array.map decode_string stringTab) in 
  let pubpathTab = new_itbl "ipubpaths" (Array.map (decode_pubpath stringTab) pubpathTab) in 
  let nlpathTab = new_itbl "inlpaths" (Array.map (decode_nlpath ccuTab stringTab) nlpathTab) in 
  let simpletypTab = new_itbl "isimpletyps" (Array.map (decode_simpletyp ccuTab stringTab nlpathTab) simpletypTab) in 
  let ((ntycons,ntypars,nvals) : sizes) = sizes in 
  let data = 
    let st1 = 
     { is = Bytes.Bytestream.of_bytes phase1bytes 0 (Bytes.length phase1bytes); 
       iccus=  ccuTab; 
       iilscope= ilscope;
       itycons=new_osgn_inmap "itycons" ntycons; 
       itypars=new_osgn_inmap "itypars" ntypars; 
       ivals=new_osgn_inmap "ivals" nvals;
       istrings = stringTab;
       ipubpaths = pubpathTab;
       inlpaths = nlpathTab;
       isimpletyps = simpletypTab; } in
    let res = u st1 in
    check ilscope st1.itycons;
    check ilscope st1.ivals;
    check ilscope st1.itypars;
    res in 
  {ie_raw=data; ie_ccus=Array.to_list ccuTab.itbl_rows }
    

(*=========================================================================*)
(* PART II *)
(*=========================================================================*)

(*---------------------------------------------------------------------------
 * Pickle/unpickle for Abstract IL data, up to IL instructions 
 *------------------------------------------------------------------------- *)

open Ast
open Il
open Ilx

let ppubkey x st = match x with 
  | PublicKey b      -> p_byte 0 st; p_bytes b st
  | PublicKeyToken b -> p_byte 1 st; p_bytes b st
let upubkey st = 
  let tag = u_byte st in match tag with
  | 0 -> u_bytes st |> (fun b -> PublicKey b) 
  | 1 -> u_bytes st |> (fun b -> PublicKeyToken b) 
  | _ -> pickle_failwith "upubkey"

let pversion x st = p_tup4 p_uint16 p_uint16 p_uint16 p_uint16 x st
let uversion st = u_tup4 u_uint16 u_uint16 u_uint16 u_uint16 st

let pmodref = p_wrap (fun x -> (x.modulRefName,x.modulRefNoMetadata,x.modulRefHash)) (p_tup3 p_string p_bool (p_option p_bytes))
let umodref = u_wrap (fun (a,b,c) -> {modulRefName=a; modulRefNoMetadata=b; modulRefHash=c}) (u_tup3 u_string u_bool (u_option u_bytes))

let passref x st =
  ptup6 p_string (p_option p_bytes) (p_option ppubkey) p_bool (p_option pversion) (p_option p_string)
  ( x.assemRefName,
    x.assemRefHash,
    x.assemRefPublicKeyInfo,
    x.assemRefRetargetable,
    x.assemRefVersion,
    x.assemRefLocale) st
let uassref st =
  let a,b,c,d,e,f = utup6 u_string (u_option u_bytes) (u_option upubkey) u_bool (u_option uversion) (u_option u_string) st in 
  { assemRefName=a;
    assemRefHash=b;
    assemRefPublicKeyInfo=c;
    assemRefRetargetable=d;
    assemRefVersion=e;
    assemRefLocale=f }

let pscoref x st = 
  match x with 
  | ScopeRef_local         -> p_byte 0 st; p_void st
  | ScopeRef_module mref   -> p_byte 1 st; pmodref mref st
  | ScopeRef_assembly aref -> p_byte 2 st; passref aref st
let uscoref st = 
  (* IL scope references are rescoped as they are unpickled.  This means *)
  (* the pickler accepts IL fragments containing ScopeRef_module and ScopeRef_local, as we adjust *)
  (* these to be absolute scope references during unpickling.  *)
  let res = 
    let tag = u_byte st in match tag with
    | 0 -> u_void   st |> (fun () -> ScopeRef_local)
    | 1 -> umodref st |> (fun mref -> ScopeRef_module mref)
    | 2 -> uassref st |> (fun aref -> ScopeRef_assembly aref)
    | _ -> pickle_failwith "uscoref"   in
  rescope_scoref st.iilscope res 

let pbasic_callconv x st = 
  p_byte (match x with 
  | CC_default -> 0
  | CC_cdecl  -> 1
  | CC_stdcall -> 2
  | CC_thiscall -> 3
  | CC_fastcall -> 4
  | CC_vararg -> 5) st
let ubasic_callconv st = 
  match u_byte st with 
  | 0 -> CC_default 
  | 1 -> CC_cdecl  
  | 2 -> CC_stdcall 
  | 3 -> CC_thiscall 
  | 4 -> CC_fastcall 
  | 5 -> CC_vararg
  | _ -> pickle_failwith "ubasic_callconv"

let phasthis x st = 
  p_byte (match x with 
  | CC_instance -> 0
  | CC_instance_explicit -> 1
  | CC_static -> 2) st
let uhasthis st = 
  match u_byte st with 
  | 0 -> CC_instance 
  | 1 -> CC_instance_explicit 
  | 2 -> CC_static 
  | _ -> pickle_failwith "uhasthis"

let pcallconv (Callconv(x,y)) st = p_tup2 phasthis pbasic_callconv (x,y) st
let ucallconv st = let a,b = u_tup2 uhasthis ubasic_callconv st in Callconv(a,b)

let piltref x st = p_tup3 pscoref pstrings p_string (x.trefScope,x.trefNested,x.trefName) st
let uiltref st = let a,b,c = u_tup3 uscoref ustrings u_string st in {trefScope=a; trefNested=b; trefName=c}


let parray_shape = p_wrap (fun (ArrayShape x) -> x) (p_list (p_tup2 (p_option p_int32) (p_option p_int32)))
let uarray_shape = u_wrap (fun x -> ArrayShape x) (u_list (u_tup2 (u_option u_int32) (u_option u_int32)))

let fill_piltyp,piltyp = p_hole()
let fill_uiltyp,uiltyp = u_hole()

let piltyps = (p_list piltyp)
let uiltyps = (u_list uiltyp)

let pcallsig = p_wrap (fun x -> (x.callsigCallconv,x.callsigArgs,x.callsigReturn)) (p_tup3 pcallconv piltyps piltyp)
let ucallsig = u_wrap (fun (a,b,c) -> {callsigCallconv=a; callsigArgs=b; callsigReturn=c}) (u_tup3 ucallconv uiltyps uiltyp)

let piltspec = p_wrap (fun { tspecTypeRef=a; tspecInst=b } -> (a,b)) (p_tup2 piltref piltyps)
let uiltspec = u_wrap (fun (a,b) -> { tspecTypeRef=a; tspecInst=b }) (u_tup2 uiltref uiltyps)

let _ = fill_piltyp (fun ty st ->
  match ty with 
  | Type_void             -> p_byte 0 st; p_void st
  | Type_array (shape,ty) -> p_byte 1 st; p_tup2 parray_shape piltyp (shape,ty) st
  | Type_value tspec      -> p_byte 2 st; piltspec tspec st
  | Type_boxed tspec      -> p_byte 3 st; piltspec tspec st
  | Type_ptr ty           -> p_byte 4 st; piltyp ty st
  | Type_byref ty         -> p_byte 5 st; piltyp ty st
  | Type_fptr csig        -> p_byte 6 st; pcallsig csig st
  | Type_tyvar n          -> p_byte 7 st; p_uint16 n st
  | Type_modified (req,tref,ty) -> p_byte 8 st; p_tup3 p_bool piltref piltyp (req,tref,ty) st
  | Type_other(e) when Ilx.is_ilx_ext_typ e -> 
      begin match Ilx.dest_ilx_ext_typ e with 
      | Ilx.EType_erasable_array(shape,ty) -> p_byte 9 st; p_tup2 parray_shape piltyp (shape,ty) st
      end
  | Type_other _ -> pickle_failwith "piltyp")
let _ = fill_uiltyp (fun st ->
  let tag = u_byte st in match tag with
  | 0 -> u_void st |> (fun () -> Type_void)
  | 1 -> u_tup2 uarray_shape uiltyp  st |> (fun (arr,ty) -> Type_array (arr,ty))
  | 2 -> uiltspec st |> (fun x -> Type_value x)
  | 3 -> uiltspec st |> (fun x -> Type_boxed x)
  | 4 -> uiltyp st |> (fun x -> Type_ptr x)
  | 5 -> uiltyp st |> (fun x -> Type_byref x)
  | 6 -> ucallsig st |> (fun x -> Type_fptr x)
  | 7 -> u_uint16 st |> (fun x -> Type_tyvar x)
  | 8 -> u_tup3 u_bool uiltref uiltyp  st |> (fun (req,tref,ty) -> Type_modified (req,tref,ty))
  | 9 -> u_tup2 uarray_shape uiltyp  st |> (fun (shape,ty) -> Ilx.mk_array_ty_old (shape,ty))
  | _ -> pickle_failwith "uiltyp")


let p_ilmref x st = ptup6 piltref pcallconv p_int p_string piltyps piltyp (x.mrefParent,x.mrefCallconv,x.mrefArity,x.mrefName,x.mrefArgs,x.mrefReturn) st
let u_ilmref st = let x1,x2,x3,x4,x5,x6 = utup6 uiltref ucallconv u_int u_string uiltyps uiltyp st in {mrefParent=x1;mrefCallconv=x2;mrefArity=x3;mrefName=x4;mrefArgs=x5;mrefReturn=x6}

let pilfref x st = p_tup3 piltref p_string piltyp (x.frefParent,x.frefName,x.frefType) st
let uilfref st = let x1,x2,x3 = u_tup3 uiltref u_string uiltyp st in {frefParent=x1;frefName=x2;frefType=x3}

let pilmspec x st = p_tup3 p_ilmref piltyp piltyps (dest_mspec x) st
let uilmspec st = let x1,x2,x3 = u_tup3 u_ilmref uiltyp uiltyps st in mk_mref_mspec_in_typ(x1,x2,x3)

let pilfspec x st = p_tup2 pilfref piltyp (x.fspecFieldRef,x.fspecEnclosingType) st
let uilfspec st = let x1,x2 = u_tup2 uilfref uiltyp st in {fspecFieldRef=x1;fspecEnclosingType=x2}

let pbasic_type x st = p_int (match x with DT_R -> 0 | DT_I1 -> 1 | DT_U1 -> 2 | DT_I2 -> 3 | DT_U2 -> 4 | DT_I4 -> 5 | DT_U4 -> 6 | DT_I8 -> 7 | DT_U8 -> 8 | DT_R4 -> 9 | DT_R8 -> 10 | DT_I -> 11 | DT_U -> 12 | DT_REF -> 13) st
let ubasic_type st = (match u_int st with  0 -> DT_R | 1 -> DT_I1 | 2 -> DT_U1 | 3 -> DT_I2 | 4 -> DT_U2 | 5 -> DT_I4 | 6 -> DT_U4 | 7 -> DT_I8 | 8 -> DT_U8 | 9 -> DT_R4 | 10 -> DT_R8 | 11 -> DT_I | 12 -> DT_U | 13 -> DT_REF | _ -> pickle_failwith "ubasic_type" )
  
let pldtoken_info x st = 
  match x with 
  | Token_type ty -> p_byte 0 st; piltyp ty st
  | Token_method x -> p_byte 1 st; pilmspec x st
  | Token_field x -> p_byte 2 st; pilfspec x st

let uldtoken_info st = 
  let tag = u_byte st in match tag with
  | 0 -> uiltyp st |> (fun x -> Token_type x)
  | 1 -> uilmspec st |> (fun x -> Token_method x)
  | 2 -> uilfspec st |> (fun x -> Token_field x)
  | _ -> pickle_failwith "uldtoken_info"
  
let pldc_info x st = 
  match x with 
  | NUM_I4 x -> p_byte 0 st; p_int32 x st
  | NUM_I8 x -> p_byte 1 st; p_int64 x st
  | NUM_R4 x -> p_byte 2 st; p_single x st
  | NUM_R8 x -> p_byte 3 st; p_ieee64 x st

let uldc_info st = 
  let tag = u_byte st in match tag with
  | 0 -> u_int32 st |> (fun x -> NUM_I4 x)
  | 1 -> u_int64 st |> (fun x -> NUM_I8 x)
  | 2 -> u_single st |> (fun x -> NUM_R4 x)
  | 3 -> u_ieee64 st |> (fun x -> NUM_R8 x)
  | _ -> pickle_failwith "uldtoken_info"
  
let palignment x st = p_int (match x with Aligned -> 0 | Unaligned_1 -> 1 | Unaligned_2 -> 2 | Unaligned_4 -> 3) st
let ualignment st = (match u_int st with  0 -> Aligned | 1 -> Unaligned_1 | 2 -> Unaligned_2 | 3 -> Unaligned_4 | _ -> pickle_failwith "ualignment" )
  
let pvolatility x st = p_int (match x with Volatile -> 0 | Nonvolatile -> 1) st
let uvolatility st = (match u_int st with  0 -> Volatile | 1 -> Nonvolatile | _ -> pickle_failwith "uvolatility" )
  
let preadonly x st = p_int (match x with ReadonlyAddress -> 0 | NormalAddress -> 1) st
let ureadonly st = (match u_int st with  0 -> ReadonlyAddress | 1 -> NormalAddress | _ -> pickle_failwith "ureadonly" )
  
let ptailness x st = p_int (match x with Tailcall -> 0 | Normalcall -> 1) st
let utailness st = (match u_int st with  0 -> Tailcall | 1 -> Normalcall | _ -> pickle_failwith "utailness" )
  
let pvarargs = p_option piltyps
let uvarargs = u_option uiltyps
  
let itag_nop         = 0 
let itag_break       = 1 
let itag_ldarg       = 2 
let itag_ldloc       = 3
let itag_stloc       = 4 
let itag_ldnull      = 5 
let itag_ldc         = 6 
let itag_dup           = 7 
let itag_pop           = 8 
let itag_jmp           = 9 
let itag_call          = 10 
let itag_calli         = 11 
let itag_ret           = 12 
let itag_br            = 13 
let itag_brfalse       = 14 
let itag_brtrue        = 15 
let itag_beq           = 16 
let itag_bge           = 17 
let itag_bgt           = 18 
let itag_ble           = 19 
let itag_blt           = 20 
let itag_bne_un        = 21 
let itag_bge_un        = 22 
let itag_bgt_un        = 23 
let itag_ble_un        = 24 
let itag_blt_un        = 25 
let itag_switch        = 26 
let itag_ldind         = 27
let itag_stind         = 28
let itag_add           = 29
let itag_sub           = 30 
let itag_mul           = 31
let itag_div           = 32 
let itag_div_un        = 33 
let itag_rem           = 34 
let itag_rem_un        = 35 
let itag_and           = 36 
let itag_or            = 37 
let itag_xor           = 38 
let itag_shl           = 39 
let itag_shr           = 40 
let itag_shr_un        = 41 
let itag_neg           = 42 
let itag_not           = 43 
let itag_conv       = 44
let itag_conv_un     = 45 
let itag_conv_ovf   = 46
let itag_conv_ovf_un   = 47
let itag_callvirt      = 48 
let itag_cpobj         = 49 
let itag_ldobj         = 50 
let itag_ldstr         = 51 
let itag_newobj        = 52 
let itag_castclass     = 53 
let itag_isinst        = 54 
let itag_unbox         = 55 
let itag_throw         = 56 
let itag_ldfld         = 57 
let itag_ldflda        = 58 
let itag_stfld         = 59 
let itag_ldsfld        = 60 
let itag_ldsflda       = 61 
let itag_stsfld        = 62 
let itag_stobj         = 63 
let itag_box           = 64 
let itag_newarr        = 65 
let itag_ldlen         = 66 
let itag_ldelema       = 67 
let itag_ldelem     = 68
let itag_stelem      = 69 
let itag_refanyval     = 70 
let itag_ckfinite      = 71 
let itag_mkrefany      = 72 
let itag_ldtoken       = 73 
let itag_add_ovf       = 74 
let itag_add_ovf_un    = 75 
let itag_mul_ovf       = 76 
let itag_mul_ovf_un    = 77 
let itag_sub_ovf       = 78 
let itag_sub_ovf_un    = 79 
let itag_endfinally    = 80 
let itag_leave         = 81 
let itag_arglist        = 82
let itag_ceq        = 83
let itag_cgt        = 84
let itag_cgt_un        = 85
let itag_clt        = 86
let itag_clt_un        = 87
let itag_ldftn        = 88 
let itag_ldvirtftn    = 89 
let itag_ldarga      = 90 
let itag_starg       = 91 
let itag_ldloca      = 92 
let itag_localloc     = 93 
let itag_endfilter    = 94 
let itag_unaligned   = 95 
let itag_volatile    = 96 
let itag_constrained    = 97
let itag_readonly    = 98
let itag_tail        = 99 
let itag_initobj             = 100
let itag_cpblk       = 101
let itag_initblk             = 102
let itag_rethrow             = 103 
let itag_sizeof      = 104
let itag_refanytype   = 105
let itag_ldelem_any = 106
let itag_stelem_any = 107
let itag_unbox_any = 108
let itag_ldunit = 109
let itag_stelem_any_erasable = 110
let itag_ldelem_any_erasable = 111
let itag_newarr_any_erasable = 112
let itag_ldlen_multi = 113
let itag_callconstrained      = 114 
let itag_ilzero = 115

let simple_instrs = 
 [
  itag_ret,              (I_ret);
  itag_add,              (I_arith AI_add);
  itag_add_ovf,        (I_arith AI_add_ovf);
  itag_add_ovf_un,   (I_arith AI_add_ovf_un);
  itag_and,              (I_arith AI_and);  
  itag_div,              (I_arith AI_div); 
  itag_div_un,         (I_arith AI_div_un);
  itag_ceq,              (I_arith AI_ceq);  
  itag_cgt,              (I_arith AI_cgt );
  itag_cgt_un,         (I_arith AI_cgt_un);
  itag_clt,              (I_arith AI_clt);
  itag_clt_un,         (I_arith AI_clt_un);
  itag_mul,   (I_arith AI_mul  );
  itag_mul_ovf,   (I_arith AI_mul_ovf);
  itag_mul_ovf_un,   (I_arith AI_mul_ovf_un);
  itag_rem,   (I_arith AI_rem  );
  itag_rem_un,   (I_arith AI_rem_un ); 
  itag_shl,   (I_arith AI_shl ); 
  itag_shr,   (I_arith AI_shr ); 
  itag_shr_un,   (I_arith AI_shr_un);
  itag_sub,   (I_arith AI_sub  );
  itag_sub_ovf,   (I_arith AI_sub_ovf);
  itag_sub_ovf_un,   (I_arith AI_sub_ovf_un); 
  itag_xor,   (I_arith AI_xor);  
  itag_or,   (I_arith AI_or);     
  itag_neg,   (I_arith AI_neg);     
  itag_not,   (I_arith AI_not);     
  itag_ldnull,   (I_arith AI_ldnull);   
  itag_dup,   (I_arith AI_dup);   
  itag_pop,   (I_arith AI_pop);
  itag_ckfinite,   (I_arith AI_ckfinite);
  itag_nop,   (I_arith AI_nop);
  itag_break,   (I_break);
  itag_arglist,   (I_arglist);
  itag_endfilter,   (I_endfilter);
  itag_endfinally,   I_endfinally;
  itag_refanytype,   (I_refanytype);
  itag_localloc,   (I_localloc);
  itag_throw,   (I_throw);
  itag_ldlen,   (I_ldlen);
  itag_rethrow,       (I_rethrow);
];;

let encode_table = Hashtbl.create 300;;
let _ = List.iter (fun (icode,i) -> Hashtbl.add encode_table i icode) simple_instrs;;
let encode_instr si = Hashtbl.find encode_table si
let is_noarg_instr s = Hashtbl.mem encode_table s

let decoders = 
 [ itag_ldarg, (u_uint16 >>  (fun x -> I_ldarg x));
   itag_starg, (u_uint16 >>  (fun x -> I_starg x));
   itag_ldarga, (u_uint16 >>  (fun x -> I_ldarga x));
   itag_stloc, (u_uint16 >>  (fun x -> I_stloc x));
   itag_ldloc, (u_uint16 >>  (fun x -> I_ldloc x));
   itag_ldloca, (u_uint16 >>  (fun x -> I_ldloca x)); 
   itag_stind, (u_tup3 ualignment uvolatility ubasic_type) >> (fun (a,b,c) -> I_stind (a,b,c));
   itag_ldind, (u_tup3 ualignment uvolatility ubasic_type) >> (fun (a,b,c) -> I_ldind (a,b,c));
   itag_cpblk, (u_tup2 ualignment uvolatility) >> (fun (a,b) -> I_cpblk (a,b));
   itag_initblk, (u_tup2 ualignment uvolatility) >> (fun (a,b) -> I_initblk (a,b));
   itag_call, (u_tup3 utailness uilmspec uvarargs) >> (fun (a,b,c) -> I_call (a,b,c));
   itag_callvirt, (u_tup3 utailness uilmspec uvarargs) >> (fun (a,b,c) -> I_callvirt (a,b,c));
   itag_callconstrained, (u_tup4 utailness uiltyp uilmspec uvarargs) >> (fun (a,b,c,d) -> I_callconstraint (a,b,c,d));
   itag_newobj, (u_tup2 uilmspec uvarargs) >> (fun (a,b) -> I_newobj (a,b));
   itag_ldftn, uilmspec >> (fun a -> I_ldftn (a));
   itag_ldvirtftn, uilmspec >> (fun a -> I_ldvirtftn (a));
   itag_calli, (u_tup3 utailness ucallsig uvarargs) >> (fun (a,b,c) -> I_calli (a,b,c));
   itag_ldc, (u_tup2 ubasic_type uldc_info) >> (fun (a,b) -> I_arith (AI_ldc (a,b)));
   itag_conv, ubasic_type >> (fun a -> I_arith (AI_conv a));
   itag_conv_ovf, ubasic_type >> (fun a -> I_arith (AI_conv_ovf a));
   itag_conv_ovf_un, ubasic_type >> (fun a -> I_arith (AI_conv_ovf_un a));
   itag_stelem, ubasic_type >> (fun a -> I_stelem a);
   itag_ldelem, ubasic_type >> (fun a -> I_ldelem a);
   itag_ldfld, (u_tup3 ualignment uvolatility uilfspec) >> (fun (a,b,c) -> I_ldfld (a,b,c));
   itag_ldflda, uilfspec >> (fun a -> I_ldflda a);
   itag_ldsfld, (u_tup2 uvolatility uilfspec) >> (fun (a,b) -> I_ldsfld (a,b));
   itag_ldsflda, uilfspec >> (fun a -> I_ldsflda a);
   itag_stfld, (u_tup3 ualignment uvolatility uilfspec) >> (fun (a,b,c) -> I_stfld (a,b,c));
   itag_stsfld, (u_tup2 uvolatility uilfspec) >> (fun (a,b) -> I_stsfld (a,b));
   itag_ldtoken, uldtoken_info >> (fun a -> I_ldtoken a);
   itag_ldstr, u_bytes >> (fun a -> I_ldstr a);
   itag_box, uiltyp >> (fun a -> I_box a);
   itag_unbox, uiltyp >> (fun a -> I_unbox a);
   itag_unbox_any, uiltyp >> (fun a -> I_unbox_any a);
   itag_newarr, u_tup2 uarray_shape uiltyp >> (fun (a,b) -> I_newarr(a,b));
   itag_stelem_any, u_tup2 uarray_shape uiltyp >> (fun (a,b) -> I_stelem_any(a,b));
   itag_ldelem_any, u_tup2 uarray_shape uiltyp >> (fun (a,b) -> I_ldelem_any(a,b));
   itag_ldelema, u_tup3 ureadonly uarray_shape uiltyp >> (fun (a,b,c) -> I_ldelema(a,b,c));
   itag_castclass, uiltyp >> (fun a -> I_castclass a);
   itag_isinst, uiltyp >> (fun a -> I_isinst a);
   itag_refanyval, uiltyp >> (fun a -> I_refanyval a);
   itag_mkrefany, uiltyp >> (fun a -> I_mkrefany a);
   itag_initobj, uiltyp >> (fun a -> I_initobj a);
   itag_initobj, uiltyp >> (fun a -> I_initobj a);
   itag_ldobj, (u_tup3 ualignment uvolatility uiltyp) >> (fun (a,b,c) -> I_ldobj (a,b,c));
   itag_stobj, (u_tup3 ualignment uvolatility uiltyp) >> (fun (a,b,c) -> I_stobj (a,b,c));
   itag_cpobj, uiltyp >> (fun a -> I_cpobj a);
   itag_sizeof, uiltyp >> (fun a -> I_sizeof a);
   itag_stelem_any_erasable, u_tup2 uarray_shape uiltyp >> (fun (a,b) -> I_other (Ilx.mk_ilx_ext_instr (Ilx.EI_stelem_any_erasable (a,b))));
   itag_ldelem_any_erasable, u_tup2 uarray_shape uiltyp >> (fun (a,b) -> I_other (Ilx.mk_ilx_ext_instr (Ilx.EI_ldelem_any_erasable (a,b))));
   itag_newarr_any_erasable, u_tup2 uarray_shape uiltyp >> (fun (a,b) -> I_other (Ilx.mk_ilx_ext_instr (Ilx.EI_newarr_erasable (a,b))));
   itag_ldlen_multi, u_tup2 u_int32 u_int32 >> (fun (a,b) -> I_other (Ilx.mk_ilx_ext_instr (Ilx.EI_ldlen_multi (a,b))));
   itag_ilzero, uiltyp >> (fun ty -> I_other (Ilx.mk_ilx_ext_instr (Ilx.EI_ilzero ty)));
   ] 

let decode_tab = 
  let tab = Array.init 256 (fun n -> (fun st -> pickle_failwith ("no decoder for instruction "^string_of_int n))) in 
  let add_instr (icode,f) =  tab.(icode) <- f in
  List.iter add_instr decoders;
  List.iter (fun (icode,mk) -> add_instr (icode,(fun _ -> mk))) simple_instrs;
  tab

let pinstr n p x st = 
  p_int n st; p x st

let rec pilinstr x st =
  match x with
  | si when is_noarg_instr si -> pinstr (encode_instr si) p_unit () st
  | I_leave _ | I_brcmp _ | I_br _ | I_switch _ -> pickle_failwith "pilinstr: cannot encode branches"
  | I_seqpoint s ->   ()
  | I_call      (tl,mspec,varargs) -> pinstr itag_call (p_tup3 ptailness pilmspec pvarargs) (tl,mspec,varargs) st;
  | I_callvirt  (tl,mspec,varargs) -> pinstr itag_callvirt (p_tup3 ptailness pilmspec pvarargs) (tl,mspec,varargs) st;
  | I_callconstraint    (tl,ty,mspec,varargs)   -> pinstr itag_callconstrained (p_tup4 ptailness piltyp pilmspec pvarargs) (tl,ty,mspec,varargs) st;
  | I_newobj    (mspec,varargs) -> pinstr itag_newobj (p_tup2 pilmspec pvarargs) (mspec,varargs) st;
  | I_ldftn     mspec   ->  pinstr itag_ldftn pilmspec mspec st;
  | I_ldvirtftn mspec   -> pinstr itag_ldvirtftn pilmspec mspec st;
  | I_calli (a,b,c)     ->  pinstr itag_calli (p_tup3 ptailness pcallsig pvarargs) (a,b,c) st;
  | I_ldarg x ->  pinstr itag_ldarg p_uint16 x st
  | I_starg x ->  pinstr itag_starg p_uint16 x st
  | I_ldarga x ->  pinstr itag_ldarga p_uint16 x st
  | I_ldloc x ->  pinstr itag_ldloc p_uint16 x st
  | I_stloc x ->  pinstr itag_stloc p_uint16 x st
  | I_ldloca x ->  pinstr itag_ldloca p_uint16 x st
  | I_cpblk     (al,vol) -> pinstr itag_cpblk (p_tup2 palignment pvolatility) (al,vol) st
  | I_initblk   (al,vol) -> pinstr itag_initblk (p_tup2 palignment pvolatility) (al,vol) st
  | I_arith (AI_ldc (a,b)) -> pinstr itag_ldc (p_tup2 pbasic_type pldc_info) (a,b) st
  | I_arith (AI_conv a) -> pinstr itag_conv pbasic_type a st
  | I_arith (AI_conv_ovf a) -> pinstr itag_conv_ovf pbasic_type a st
  | I_arith (AI_conv_ovf_un a) -> pinstr itag_conv_ovf_un pbasic_type a st
  | I_ldind (a,b,c) -> pinstr itag_ldind (p_tup3 palignment pvolatility pbasic_type) (a,b,c) st
  | I_stind (a,b,c) -> pinstr itag_stind (p_tup3 palignment pvolatility pbasic_type) (a,b,c) st
  | I_stelem a  -> pinstr itag_stelem pbasic_type a st 
  | I_ldelem a  -> pinstr itag_ldelem pbasic_type a st 
  | I_ldfld(a,b,c) -> pinstr itag_ldfld (p_tup3 palignment pvolatility pilfspec) (a,b,c) st
  | I_ldflda(c) -> pinstr itag_ldflda pilfspec c st
  | I_ldsfld(a,b) -> pinstr itag_ldsfld (p_tup2 pvolatility pilfspec) (a,b) st
  | I_ldsflda(a) -> pinstr itag_ldsflda pilfspec a st
  | I_stfld(a,b,c) -> pinstr itag_stfld (p_tup3 palignment pvolatility pilfspec) (a,b,c) st
  | I_stsfld(a,b) -> pinstr itag_stsfld (p_tup2 pvolatility pilfspec) (a,b) st
  | I_ldtoken  tok -> pinstr itag_ldtoken pldtoken_info tok st
  | I_ldstr     s       -> pinstr itag_ldstr p_bytes s st
  | I_box  ty   -> pinstr itag_box piltyp ty st
  | I_unbox  ty -> pinstr itag_unbox piltyp ty st
  | I_unbox_any  ty     -> pinstr itag_unbox_any piltyp ty st
  | I_newarr(a,b)       -> pinstr itag_newarr (p_tup2 parray_shape piltyp) (a,b) st
  | I_stelem_any(a,b)   -> pinstr itag_stelem_any (p_tup2 parray_shape piltyp) (a,b) st
  | I_ldelem_any(a,b)   -> pinstr itag_ldelem_any (p_tup2 parray_shape piltyp) (a,b) st
  | I_ldelema(a,b,c)    -> pinstr itag_ldelema (p_tup3 preadonly parray_shape piltyp) (a,b,c) st
  | I_castclass  ty     -> pinstr itag_castclass piltyp ty st
  | I_isinst  ty        -> pinstr itag_isinst piltyp ty st
  | I_refanyval  ty     -> pinstr itag_refanyval piltyp ty st
  | I_mkrefany  ty      -> pinstr itag_mkrefany piltyp ty st
  | I_initobj  ty       -> pinstr itag_initobj piltyp ty st
  | I_ldobj(a,b,c)      -> pinstr itag_ldobj (p_tup3 palignment pvolatility piltyp) (a,b,c) st
  | I_stobj(a,b,c)      -> pinstr itag_stobj (p_tup3 palignment pvolatility piltyp) (a,b,c) st
  | I_cpobj  ty         -> pinstr itag_cpobj piltyp ty st
  | I_sizeof  ty        -> pinstr itag_sizeof piltyp ty st
  | I_other e when Ilx.is_ilx_ext_instr e -> 
      begin match (Ilx.dest_ilx_ext_instr e) with 
      | Ilx.EI_ldftn_then_call (mr1,(tl,mr2,varargs)) -> pickle_failwith "cannot pickle instructions internal to ILX"
      | Ilx.EI_ld_instance_ftn_then_newobj (mr1,_,(mr2,varargs)) -> pickle_failwith "cannot pickle instructions internal to ILX"
      | Ilx.EI_stelem_any_erasable (a,b) ->  pinstr itag_stelem_any_erasable (p_tup2 parray_shape piltyp) (a,b) st
      | Ilx.EI_ldelem_any_erasable (a,b) -> pinstr itag_ldelem_any_erasable (p_tup2 parray_shape piltyp) (a,b) st
      | Ilx.EI_newarr_erasable (a,b) -> pinstr itag_newarr_any_erasable (p_tup2 parray_shape piltyp) (a,b) st
      | Ilx.EI_ldlen_multi (n,m) -> pinstr itag_ldlen_multi (p_tup2 p_int32 p_int32) (n,m) st
      | Ilx.EI_ilzero (a) -> pinstr itag_ilzero piltyp a st
      |  _ -> pickle_failwith "an ILX instruction cannot be emitted"
      end
  |  _ -> pickle_failwith "an IL instruction cannot be emitted"


let uilinstr st = 
  let n = u_int st in 
  decode_tab.(n) st

  

(*---------------------------------------------------------------------------
 * Pickle/unpickle for F# types and module signatures
 *------------------------------------------------------------------------- *)

let pPmap pk pv = p_wrap pmap_to_list (p_list (p_tup2 pk pv))
let uPmap uk uv = u_wrap pmap_of_list (u_list (u_tup2 uk uv))

let p_namemap p = pPmap p_string p
let u_namemap u = uPmap u_string u

(*let pnamemmap p = p_namemap (p_list p)
let unamemmap u = u_namemap (u_list u) *)

let pimmutable_ref p = p_wrap (!) p
let uimmutable_ref u = u_wrap (ref) u

let ppos (x: pos) st = p_tup2 p_int p_int (dest_pos x) st
let upos st = let a = u_int st in let b = u_int st in mk_pos a b

let prange (x: range) st = p_tup3 p_string ppos ppos (dest_range x) st
let urange st = let a = u_string st in let b = upos st in let c = upos st in mk_range a b c

(* Most ranges (e.g. on optimization expressions) can be elided from stored data *)
let dummyRange = range0 
let pdrange : range pickler   = fun x st -> ()
let udrange : range unpickler = fun st -> dummyRange

let pident (x: ident) st = p_tup2 p_string prange (x.idText,x.idRange) st
let uident st = let a = u_string st in let b = urange st in ident(a,b)


let pnonlocal_item_ref () {nlr_nlpath=a;nlr_item=b} st =
  pnlpath a st; p_string b st
let unonlocal_item_ref () st = 
  let a = unlpath st in let b = u_string st in 
  mk_nlr a b
  
let plocal_item_ref ctxt tab st = posgn_ref ctxt tab st
let ulocal_item_ref tab st = uosgn_ref tab st

let rec pitem_ref ctxt tab x st = 
  match x with 
  | Ref_private(x) -> p_byte 0 st; plocal_item_ref ctxt tab x st
  | Ref_nonlocal(x) -> p_byte 1 st; pnonlocal_item_ref () x st
let uitem_ref tab st = 
  let tag = u_byte st in match tag with
  | 0 -> ulocal_item_ref tab st |> (fun x -> Ref_private x)
  | 1 -> unonlocal_item_ref () st |> (fun x -> Ref_nonlocal x)
  | _ -> pickle_failwith "uitem_ref"
    
let ( !! ) (x: 'a Lazy.t) = Lazy.force x

let p_tcref ctxt (x: tycon_ref) st = pitem_ref ctxt st.otycons x st
let u_tcref st = uitem_ref st.itycons st

let p_ucref (UCRef(a,b)) st = p_tup2 (p_tcref "ucref") p_string (a,b) st
let u_ucref st  = let a,b = u_tup2 u_tcref u_string st in UCRef(a,b)

let prfref (RFRef(a,b)) st = p_tup2 (p_tcref "rfref") p_string (a,b) st
let urfref st = let a,b = u_tup2 u_tcref u_string st in RFRef(a,b)

let p_vref ctxt x st = pitem_ref ctxt st.ovals  x st
let u_vref st = uitem_ref st.ivals st

let pvrefs ctxt = p_list (p_vref ctxt) 
let uvrefs = u_list u_vref 

let ptpref x st = plocal_item_ref "typar" st.otypars  x st
let utpref st = ulocal_item_ref st.itypars st

let fill_ptyp,p_typ = p_hole()
let fill_utyp,u_typ = u_hole()

let ptyps = (p_list p_typ);;
let utyps = (u_list u_typ);;

let ptypar_constraint_typ x st = 
    match x with 
    | TTyparSubtypeConstraintFromIL (a,b) -> p_byte 0 st; p_tup2 pscoref piltyp (a,b) st
    | TTyparSubtypeConstraintFromFS(a)    -> p_byte 1 st; p_typ a st

let utypar_constraint_typ st = 
    let tag = u_byte st in match tag with
    | 0 -> u_tup2 uscoref uiltyp st |> (fun (a,b) -> TTyparSubtypeConstraintFromIL (a,b) ) 
    | 1 -> u_typ st                 |> (fun a     -> TTyparSubtypeConstraintFromFS(a))
    | _ -> pickle_failwith "utypar_constraint_typ" 

let pmemberkind x st = 
    p_byte (match x with 
    | MemberKindMember -> 0
    | MemberKindPropertyGet  -> 1
    | MemberKindPropertySet -> 2
    | MemberKindConstructor -> 3
    | MemberKindClassConstructor -> 4
    | MemberKindPropertyGetSet -> pickle_failwith "pickling: MemberKindPropertyGetSet only expected in parse trees") st

let umemberkind st = 
    match u_byte st with 
    | 0 -> MemberKindMember 
    | 1 -> MemberKindPropertyGet  
    | 2 -> MemberKindPropertySet 
    | 3 -> MemberKindConstructor
    | 4 -> MemberKindClassConstructor
    | _ -> pickle_failwith "umemberkind"

let pmemberflags x st = 
  ptup7 (p_option p_bytes) p_bool p_bool p_bool p_bool p_bool pmemberkind (x.memFlagsOverloadQualifier, 
                                                                    x.memFlagsInstance, 
                                                                    x.memFlagsVirtual, 
                                                                    x.memFlagsAbstract, 
                                                                    x.memFlagsOverride, 
                                                                    x.memFlagsFinal, 
                                                                    x.memFlagsKind) st
let umemberflags st = 
  let x1,x2,x3,x4,x5,x6,x7 = utup7 (u_option u_bytes) u_bool u_bool u_bool u_bool u_bool umemberkind st in 
  { memFlagsOverloadQualifier=x1;
    memFlagsInstance=x2;
    memFlagsVirtual=x3;
    memFlagsAbstract=x4;
    memFlagsOverride=x5;
    memFlagsFinal=x6;
    memFlagsKind=x7}

let ptrait (TTrait (a,b,c,d,e)) st  = 
    p_tup5 ptyps p_string pmemberflags ptyps p_typ (a,b,c,d,e) st
let utrait st = 
    let a,b,c,d,e = u_tup5 utyps u_string umemberflags utyps u_typ st in
    TTrait(a,b,c,d,e)

let ptypar_constraint x st = 
  match x with 
  | TTyparCoercesToType (a,m)                 -> p_byte 0 st; p_tup2 ptypar_constraint_typ prange (a,m) st
  | TTyparMayResolveMemberConstraint(traitInfo,m) -> p_byte 1 st; p_tup2 ptrait prange (traitInfo,m) st
  | TTyparDefaultsToType(_,rty,m)               -> p_byte 2 st; p_tup2 p_typ prange (rty,m) st
  | TTyparSupportsNull(m)                     -> p_byte 3 st; prange m st
  | TTyparIsNotNullableValueType(m)           -> p_byte 4 st; prange m st
  | TTyparIsReferenceType(m)                  -> p_byte 5 st; prange m st
  | TTyparRequiresDefaultConstructor(m)       -> p_byte 6 st; prange m st
  | TTyparSimpleChoice(tys,m)                 -> p_byte 7 st; p_tup2 ptyps prange (tys,m) st
  | TTyparIsEnum(ty,m)                        -> p_byte 8 st; p_typ ty st; prange m st
  | TTyparIsDelegate(aty,bty,m)               -> p_byte 9 st; p_typ aty st; p_typ bty st; prange m st
let utypar_constraint st = 
  let tag = u_byte st in match tag with
  | 0 -> u_tup2 utypar_constraint_typ urange st |> (fun (a,b)   -> (fun ridx -> TTyparCoercesToType (a,b) ))
  | 1 -> u_tup2 utrait urange st |> (fun (traitInfo,f) -> (fun ridx -> TTyparMayResolveMemberConstraint(traitInfo,f)))
  | 2 -> u_tup2 u_typ urange                  st |> (fun (a,b)   -> (fun ridx -> TTyparDefaultsToType(ridx,a,b)))
  | 3 -> urange                             st |> (fun (a)     -> (fun ridx -> TTyparSupportsNull(a)))
  | 4 -> urange                             st |> (fun (a)     -> (fun ridx -> TTyparIsNotNullableValueType(a)))
  | 5 -> urange                             st |> (fun (a)     -> (fun ridx -> TTyparIsReferenceType(a)))
  | 6 -> urange                             st |> (fun (a)     -> (fun ridx -> TTyparRequiresDefaultConstructor(a)))
  | 7 -> u_tup2 utyps urange                st |> (fun (a,b)   -> (fun ridx -> TTyparSimpleChoice(a,b)))
  | 8 -> u_tup2 u_typ urange                st |> (fun (a,b)   -> (fun ridx -> TTyparIsEnum(a,b)))
  | 9 -> u_tup3 u_typ u_typ urange          st |> (fun (a,b,c) -> (fun ridx -> TTyparIsDelegate(a,b,c)))
  | _ -> pickle_failwith "utypar_constraint" 

let pxmldoc = p_wrap (function XMLDoc x -> x) pstrings
let uxmldoc = u_wrap (function [] -> emptyXMLDoc | x -> XMLDoc x) ustrings

let ptypar_constraints = (p_list ptypar_constraint)
let utypar_constraints = (ulisti utypar_constraint)

let fill_pattribs,pattribs = p_hole()
let fill_uattribs,uattribs = u_hole()


let ptypar_spec_data (x:typar_spec_data) st = 
  if (rigid_of_tpdata x <> TyparRigid) then warning(Error("ptypar_spec: typar is not rigid", x.typar_id.idRange));
  if (from_error_of_tpdata x) then warning(Error("ptypar_spec: from error", x.typar_id.idRange));
  p_tup5
    pident
    pattribs 
    p_int32
    ptypar_constraints
    pxmldoc
    (x.typar_id,x.typar_attribs,x.typar_flags,x.typar_constraints,x.typar_xmldoc) st

let utypar_spec_data st = 
  let a,c,d,e,g = u_tup5 uident uattribs u_int32 utypar_constraints uxmldoc st in 
  { typar_id=a; 
    typar_stamp=new_stamp();
    typar_attribs=c;
    typar_flags=d;
    typar_constraints=e;
    typar_solution=TType_unknown; 
    typar_xmldoc=g }

let ptypar_spec x st = posgn_decl st.otypars ptypar_spec_data x st
let utypar_spec st = uosgn_decl st.itypars utypar_spec_data st 

let ptypar_specs = (p_list ptypar_spec)
let utypar_specs = (u_list utypar_spec)


let _ = fill_ptyp (fun ty st ->
   let ty = strip_tpeqns ty in
   match ty with 
   | TType_tuple l                                                 -> p_byte 0 st; ptyps l st
   | TType_app(Ref_nonlocal { nlr_nlpath=nlpath; nlr_item=item },[]) -> p_byte 1 st; psimpletyp (nlpath,item) st
   | TType_app (tc,tinst)                                          -> p_byte 2 st; p_tup2 (p_tcref "typ") ptyps (tc,tinst) st
   | TType_fun (d,r)                                               -> p_byte 3 st; p_tup2 p_typ p_typ (d,r) st
   | TType_var r                                                   -> p_byte 4 st; ptpref r st
   | TType_forall (tps,r)                                          -> p_byte 5 st; p_tup2 ptypar_specs p_typ (tps,r) st
   | TType_modul_bindings                                          -> p_byte 6 st; p_void st
   | TType_unknown                                                 -> p_byte 7 st; p_void st)
let _ = fill_utyp (fun st ->
    let tag = u_byte st in match tag with
    | 0 -> let l = utyps st                              in TType_tuple l
    | 1 -> usimpletyp st 
    | 2 -> let tc = u_tcref st in let tinst = utyps st    in TType_app (tc,tinst)
    | 3 -> let d = u_typ st    in let r = u_typ st         in TType_fun (d,r)
    (* REVIEW: ALLOCATION SORE POINT *)
    | 4 -> let r = utpref st                             in TType_var r
    | 5 -> let tps = utypar_specs st in let r = u_typ st  in TType_forall (tps,r)
    | 6 ->                                                  TType_modul_bindings
    | 7 ->                                                  TType_unknown
    | _ -> pickle_failwith "u_typ")
  

let fill_pbinds,pbinds = p_hole()
let fill_ubinds,ubinds = u_hole()

let fill_ptargets,ptargets = p_hole()
let fill_utargets,utargets = u_hole()

let fill_pexprs,pexprs = p_hole()
let fill_uexprs,uexprs = u_hole()

let fill_pconstraints,pconstraints = p_hole()
let fill_uconstraints,uconstraints = u_hole()

let fill_pval_specs,pval_specs = p_hole()
let fill_uval_specs,uval_specs = u_hole()

let pargmd (TopArgData(a,b)) st = pattribs a st; p_option pident b st
let uargmd st = let a = uattribs st in let b = u_option uident st in match a,b with [],None -> TopValData.unnamedTopArg1 | _ -> TopArgData(a,b)

let pval_arity (TopValInfo (a,args,ret)) st = 
    p_int a st; 
    p_list (p_list pargmd) args st;
    pargmd ret st
let uval_arity st = 
    let a = u_int st in 
    let b = u_list (u_list uargmd) st in
    let c = uargmd st in 
    TopValInfo (a,b,c)

let pranges = (p_option (p_tup2 prange prange)) 
let uranges = (u_option (u_tup2 urange urange))

let pistype x st = 
  match x with 
  | AsMangledNamedType str -> p_byte 0 st; p_string str st
  | AsNamedType            -> p_byte 1 st
  | Namespace              -> p_byte 2 st
let uistype st = 
  let tag = u_byte st in match tag with
  | 0 -> AsMangledNamedType (u_string st)
  | 1 -> AsNamedType  
  | 2 -> Namespace 
  | _ -> pickle_failwith "uistype"

let pcpath (CompPath(a,b)) st = p_tup2 pscoref (p_list (p_tup2 p_string pistype)) (a,b) st
let ucpath  st = let a,b = u_tup2 uscoref (u_list (u_tup2 u_string uistype)) st in (CompPath(a,b))


let rec prepr_spec x st = 
  match x with 
  | TRecdRepr fs          -> p_byte 0 st; prfield_table fs st
  | TFiniteUnionRepr x    -> p_byte 2 st; p_list punionconstr_spec (Array.to_list x.funion_constrs.uconstrs_by_index) st
  | TIlObjModelRepr (_,_,td) -> error (Failure("Generated IL types such as "^name_of_tdef td^" may not currently be exported from a DLL"))
  | TAsmRepr ilty         -> p_byte 4 st; piltyp ilty st
  | TFsObjModelRepr r     -> p_byte 5 st; ptycon_objmodel_data r st
and urepr_spec st = 
  let tag = u_byte st in match tag with
  | 0 -> urfield_table        st |> (fun fs -> TRecdRepr fs)
  | 2 -> u_list uunionconstr_spec st |> (fun x -> mk_TFiniteUnionRepr x)
  | 3 -> pickle_failwith "TIlObjModelRepr: must currently be private"
  | 4 -> uiltyp                  st |> (fun x -> TAsmRepr x)
  | 5 -> utycon_objmodel_data        st |> (fun r -> TFsObjModelRepr r)
  | _ -> pickle_failwith "urepr_spec"
  
and ptycon_objmodel_data x st = 
  if verbose then dprintf0 "ptycon_objmodel_data\n";
  p_tup3 ptycon_objmodel_kind (pvrefs "vslots") prfield_table 
    (x.tycon_objmodel_kind, x.fsobjmodel_vslots, x.fsobjmodel_rfields) st
and utycon_objmodel_data st = 
  let x1,x2,x3 = u_tup3 utycon_objmodel_kind uvrefs urfield_table st in 
  {tycon_objmodel_kind=x1; fsobjmodel_vslots=x2; fsobjmodel_rfields=x3 }
  
and punionconstr_spec x st =                     
    if verbose then dprintf0 "punionconstr_spec\n";
    ptup7 
        prfield_table p_typ p_string pident pattribs pxmldoc paccess
        (x.uconstr_rfields,x.uconstr_rty,x.uconstr_il_name,x.uconstr_id,x.uconstr_attribs,x.uconstr_xmldoc,x.uconstr_access) st
and uunionconstr_spec st = 
    let a,b,c,d,e,f,i = utup7 urfield_table u_typ u_string uident uattribs uxmldoc uaccess st in 
    {uconstr_rfields=a; uconstr_rty=b; uconstr_il_name=c; uconstr_id=d; uconstr_attribs=e;uconstr_xmldoc=f;uconstr_access=i }
    
and pexnc_spec_data x st = ptycon_spec_data x st
and uexnc_spec_data st = utycon_spec_data st 

and pexnc_repr x st =
  match x with 
  | TExnAbbrevRepr x -> p_byte 0 st; (p_tcref "exn abbrev") x st
  | TExnAsmRepr x    -> p_byte 1 st; piltref x st
  | TExnFresh x      -> p_byte 2 st; prfield_table x st
  | TExnNone         -> p_byte 3 st
and uexnc_repr st =
  let tag = u_byte st in match tag with
  | 0 -> u_tcref           st |> (fun x -> TExnAbbrevRepr x)
  | 1 -> uiltref          st |> (fun x -> TExnAsmRepr x)
  | 2 -> urfield_table st |> (fun x -> TExnFresh x)
  | 3 -> TExnNone
  | _ -> pickle_failwith "uexnc_repr"
  
and pexnc_spec x st = ptycon_spec x st
and uexnc_spec st = utycon_spec st

and paccess (TAccess n) st = p_list pcpath n st
and uaccess st = 
    match u_list ucpath st with 
    | [] -> taccessPublic (* save unnecessary allocations *)
    | res -> TAccess res

and precdfield_spec x st = 
    if verbose then dprintf0 "precdfield_spec\n";
    ptup10
      p_bool p_typ p_bool p_bool (p_option p_const) pident pattribs pattribs pxmldoc paccess 
      (x.rfield_mutable,x.rfield_type,x.rfield_static,x.rfield_secret,x.rfield_const,x.rfield_id,x.rfield_pattribs,x.rfield_fattribs,x.rfield_xmldoc,x.rfield_access) st
and urecdfield_spec st = 
    let a,c1,c2,c2b,c3,d,e1,e2,f,g = utup10 u_bool u_typ u_bool u_bool (u_option u_const) uident uattribs uattribs uxmldoc uaccess st in 
    { rfield_mutable=a;  rfield_type=c1; rfield_static=c2; rfield_secret=c2b; rfield_const=c3; rfield_id=d; rfield_pattribs=e1;rfield_fattribs=e2;rfield_xmldoc=f; rfield_access=g }

and prfield_table x st = 
  if verbose then dprintf0 "prfield_table\n";
  p_list precdfield_spec (Array.to_list x.rfields_by_index) st
and urfield_table st = mk_rfields_table (u_list urecdfield_spec st)

and ptycon_spec_data x st = 
  if verbose then dprintf0 "ptycon_spec_data\n";
  ptup14
    ptypar_specs
    pident 
    (p_option ppubpath)
    (p_tup2 paccess paccess)
    pattribs
    (p_option prepr_spec)
    (p_option p_typ)
    ptcaug
    pxmldoc
    p_bool 
    p_bool 
    (p_option pcpath)
    (plazy_immediate pmodul_typ)
    pexnc_repr 
    (x.tycon_typars,
     ident (x.tycon_name, x.tycon_range),
     x.tycon_pubpath,
     (x.tycon_access, x.tycon_repr_access),
     x.tycon_attribs,
     x.tycon_repr,
     x.tycon_abbrev,
     x.tycon_tcaug,
     x.tycon_xmldoc,
     x.tycon_prefix_display,
     x.tycon_is_modul,
     x.tycon_cpath,
     x.tycon_modul_contents,
     x.tycon_exnc_info ) st
and utycon_spec_data st = 
  let x1,x2,x3,(x4a,x4b),x6,x7,x8,x9,x10,x11,x11b,x12,x13,x14 = 
   utup14
    utypar_specs
    uident 
    (u_option upubpath)
    (u_tup2 uaccess uaccess)
    uattribs
    (u_option urepr_spec)
    (u_option u_typ) 
    utcaug 
    uxmldoc 
    u_bool 
    u_bool 
    (u_option ucpath )
    (ulazy_immediate umodul_typ) 
    uexnc_repr 
    st in 
    { tycon_typars=x1;
      tycon_stamp=new_stamp();
      tycon_name=x2.idText;
      tycon_range=x2.idRange;
      tycon_pubpath=x3;
      tycon_access=x4a;
      tycon_repr_access=x4b;
      tycon_attribs=x6;
      tycon_repr=x7;
      tycon_abbrev=x8;
      tycon_tcaug=x9;
      tycon_xmldoc=x10;
      tycon_prefix_display=x11;
      tycon_is_modul=x11b;
      tycon_cpath=x12;
      tycon_modul_contents= x13;
      tycon_exnc_info=x14;
      tycon_il_repr_cache=new_cache();  } 
and ptcaug p st = 
  if verbose then dprintf0 "ptcaug\n";
  ptup8
    (p_option (p_vref "compare"))
    (p_option (p_vref "hash"))
    (p_option (p_tup2 (p_vref "equals_obj") (p_vref "equals")))
    (p_namemap (pvrefs "adhoc")) 
    (p_list (p_tup3 p_typ p_bool prange))
    (p_option p_typ)
    p_bool
    p_bool
    (p.tcaug_compare, p.tcaug_structural_hash, p.tcaug_equals, p.tcaug_adhoc, p.tcaug_implements,p.tcaug_super,p.tcaug_closed,p.tcaug_abstract) st
and utcaug st = 
  let a,b1,b2,c,d,e,f,g = 
    utup8
      (u_option u_vref)
      (u_option u_vref)
      (u_option (u_tup2 u_vref u_vref))
      (u_namemap uvrefs)
      (u_list (u_tup3 u_typ u_bool urange)) 
      (u_option u_typ)
      u_bool 
      u_bool 
      st  in
  {tcaug_compare=a; tcaug_structural_hash=b1; tcaug_equals=b2; 
   (* only used for code generation and checking - hence don't care about the values when reading back in *)
   tcaug_hasObjectGetHashCode=false; 
   tcaug_adhoc=c; tcaug_implements=d;tcaug_super=e;tcaug_closed=f; tcaug_abstract=g}
 
and ptycon_spec x st = posgn_decl st.otycons ptycon_spec_data x st
and utycon_spec st = uosgn_decl st.itycons utycon_spec_data st 

and p_parentref x st = 
  match x with 
  | ParentNone -> p_byte 0 st
  | Parent x -> p_byte 1 st; p_tcref "parent tycon" x st
and u_parentref st = 
  let tag = u_byte st in match tag with
  | 0 -> ParentNone
  | 1 -> u_tcref st |> (fun x -> Parent x)
  | _ -> pickle_failwith "uattribkind" 

and pattribkind x st = 
  match x with 
  | ILAttrib x -> p_byte 0 st; p_ilmref x st
  | FSAttrib x -> p_byte 1 st; p_vref "attrib" x st
and uattribkind st = 
  let tag = u_byte st in match tag with
  | 0 -> u_ilmref st |> (fun x -> ILAttrib x) 
  | 1 -> u_vref st |> (fun x -> FSAttrib x)
  | _ -> pickle_failwith "uattribkind" 

and pattrib (Attrib x) st = 
    if verbose then dprintf0 "pattrib\n";
    p_tup3 pattribkind pexprs (p_list (p_tup4 p_string p_typ p_bool p_expr)) x st

and uattrib st : attrib = 
    Attrib(u_tup3 uattribkind uexprs (u_list (u_tup4 u_string u_typ u_bool u_expr)) st)



and pvspr x st = 
    if verbose then dprintf0 "pvspr\n";
    p_tup5 p_string 
        (p_tcref "vspr") pmemberflags (p_option pslotsig) p_bool 
        (x.vspr_il_name, x.vspr_apparent_parent,x.vspr_flags,x.vspr_implements_slotsig,x.vspr_implemented) st
and uvspr st = 
  let x1,x2a,x3,x4,x5 = u_tup5 u_string u_tcref umemberflags (u_option uslotsig) u_bool st in 
  { vspr_il_name=x1;
    vspr_apparent_parent=x2a;
    vspr_flags=x3;
    vspr_implements_slotsig=x4;
    vspr_implemented=x5  }

and ptycon_objmodel_kind x st = 
  match x with 
  | TTyconClass       -> p_byte 0 st; p_void st
  | TTyconInterface   -> p_byte 1 st; p_void st
  | TTyconStruct      -> p_byte 2 st; p_void st
  | TTyconDelegate ss -> p_byte 3 st; pslotsig ss st
  | TTyconEnum        -> p_byte 4 st; p_void st
and utycon_objmodel_kind st = 
  let tag = u_byte st in match tag with
  | 0 -> u_void    st |> (fun () -> TTyconClass )
  | 1 -> u_void    st |> (fun () -> TTyconInterface  )
  | 2 -> u_void    st |> (fun () -> TTyconStruct )
  | 3 -> uslotsig st |> (fun x  -> TTyconDelegate x)
  | 4 -> u_void    st |> (fun () -> TTyconEnum )
  | _ -> pickle_failwith "utycon_objmodel_kind"

and pmustinline x st = 
  p_byte (match x with 
  | PseudoValue -> 0
  | AlwaysInline  -> 1
  | OptionalInline -> 2
  | NeverInline -> 3) st
and umustinline st = 
  match u_byte st with 
  | 0 -> PseudoValue 
  | 1 -> AlwaysInline  
  | 2 -> OptionalInline 
  | 3 -> NeverInline 
  | _ -> pickle_failwith "umustinline"

and pbasethis x st = 
  p_byte (match x with 
  | BaseVal -> 0
  | CtorThisVal  -> 1
  | NormalVal -> 2
  | MemberThisVal -> 3) st
and ubasethis st = 
  match u_byte st with 
  | 0 -> BaseVal 
  | 1 -> CtorThisVal  
  | 2 -> NormalVal 
  | 3 -> MemberThisVal
  | _ -> pickle_failwith "ubasethis"

and pvrefFlags x st = 
  p_byte (match x with 
  | NormalValUse -> 0
  | CtorValUsedAsSuperInit  -> 1
  | CtorValUsedAsSelfInit  -> 2
  | VSlotDirectCall -> 3) st
and uvrefFlags st = 
  match u_byte st with 
  | 0 -> NormalValUse 
  | 1 -> CtorValUsedAsSuperInit
  | 2 -> CtorValUsedAsSelfInit
  | 3 -> VSlotDirectCall
  | _ -> pickle_failwith "uvrefFlags"

and pval_spec_data x st =
    if verbose then dprintf3 "pval_spec_data, nm = %s, stamp #%d, ty = %s\n" x.val_name x.val_stamp (DebugPrint.showType x.val_type);
    ptup12
      p_string
      pranges
      p_typ 
      p_int64 
      (p_option ppubpath) 
      (p_option pvspr) 
      pattribs 
      (p_option pval_arity)
      pxmldoc
      paccess
      p_parentref
      (p_option p_const)
    ( x.val_name,
      (* only keep range information on published values, not on optimization data *)
      (match x.val_pubpath with None -> None | Some _ -> Some(x.val_range, x.val_defn_range)),
      x.val_type,
      x.val_flags,
      x.val_pubpath,
      x.val_meminfo,
      x.val_attribs,
      x.val_arity,
      x.val_xmldoc,
      x.val_access,
      x.val_actual_parent,
      x.val_const) st
      
and uval_spec_data st =
  let x1,x1a,x2,x4,x7,x8,x9,x10,x12,x13,x13b,x14 = 
    utup12
      u_string
      uranges
      u_typ 
      u_int64
      (u_option upubpath)
      (u_option uvspr) 
      uattribs 
      (u_option uval_arity)
      uxmldoc 
      uaccess
      u_parentref
      (u_option u_const) st in 
  { val_name=x1;
    val_range=(match x1a with None -> dummyRange | Some(a,_) -> a);
    val_defn_range=(match x1a with None -> dummyRange | Some(_,b) -> b);
    val_type=x2;
    val_stamp=new_stamp();
    val_flags=x4;
    val_pubpath=x7;
    val_defn = None;
    val_meminfo=x8;
    val_attribs=x9;
    val_arity=x10;
    val_xmldoc=x12;
    val_access=x13;
    val_actual_parent=x13b;
    val_const=x14;
  }

and pval_spec x st = posgn_decl st.ovals pval_spec_data x st
and uval_spec st = uosgn_decl st.ivals uval_spec_data st 


and pmodul_spec x st = ptycon_spec x st
and umodul_spec st = utycon_spec st

and pmodul_typ (x: modul_typ) st = 
    p_tup4
      pistype
      (p_namemap pmodul_spec)
      (p_namemap pval_spec)
      (p_namemap ptycon_spec)
      (x.mtyp_kind,x.mtyp_submoduls,x.mtyp_vals,x.mtyp_tycons )
      st

and umodul_typ st = 
  let x1,x2,x3,x5 = 
    u_tup4
      uistype
      (u_namemap umodul_spec)
      (u_namemap uval_spec)
      (u_namemap utycon_spec) st in 
  { mtyp_kind=x1;
    mtyp_submoduls=x2;
    mtyp_vals=x3;
    mtyp_tycons=x5;
    mtyp_apref_cache = ref None;
    mtyp_tyconsByDemangledNameAndArity_cache = ref None;
    mtyp_tyconsByAccessNames_cache = ref None ;
    mtyp_exconsByDemangledName_cache= ref None}


(*---------------------------------------------------------------------------
 * Pickle/unpickle for F# expressions (for optimization data)
 *------------------------------------------------------------------------- *)

and p_const x st = 
  match x with 
  | TConst_bool x       -> p_byte 0  st; p_bool x st
  | TConst_int8 x       -> p_byte 1  st; p_int8 x st
  | TConst_uint8 x      -> p_byte 2  st; p_uint8 x st
  | TConst_int16 x      -> p_byte 3  st; p_int16 x st
  | TConst_uint16 x     -> p_byte 4  st; p_uint16 x st
  | TConst_int32 x      -> p_byte 5  st; p_int32 x st
  | TConst_uint32 x     -> p_byte 6  st; p_uint32 x st
  | TConst_int64 x      -> p_byte 7  st; p_int64 x st
  | TConst_uint64 x     -> p_byte 8  st; p_uint64 x st
  | TConst_nativeint x  -> p_byte 9  st; p_int64 x st
  | TConst_unativeint x -> p_byte 10 st; p_uint64 x st
  | TConst_float32 x    -> p_byte 11 st; p_single x st
  | TConst_float x      -> p_byte 12 st; p_int64 (Nums.ieee64_to_bits x) st
  | TConst_char c       -> p_byte 13 st; p_char c st
  | TConst_string s     -> p_byte 14 st; p_bytes s st
  | TConst_bigint s     -> p_byte 15 st; p_bytes s st
  | TConst_bignum s     -> p_byte 16 st; p_bytes s st
  | TConst_unit         -> p_byte 17 st; p_void st
  | TConst_default      -> p_byte 18 st; p_void st
  | TConst_decimal s    -> p_byte 19 st; p_bytes s st
and u_const st = 
  let tag = u_byte st in match tag with
  | 0 -> u_bool st |> (fun x -> TConst_bool x) 
  | 1 -> u_int8 st |> (fun x -> TConst_int8 x)
  | 2 -> u_uint8 st |> (fun x -> TConst_uint8 x)
  | 3 -> u_int16 st |> (fun x -> TConst_int16 x)
  | 4 -> u_uint16 st |> (fun x -> TConst_uint16 x)
  | 5 -> u_int32 st |> (fun x -> TConst_int32 x)
  | 6 -> u_uint32 st |> (fun x -> TConst_uint32 x)
  | 7 -> u_int64 st |> (fun x -> TConst_int64 x)
  | 8 -> u_uint64 st |> (fun x -> TConst_uint64 x)
  | 9 -> u_int64 st |> (fun x -> TConst_nativeint x)
  | 10 -> u_uint64 st |> (fun x -> TConst_unativeint x)
  | 11 -> u_single st |> (fun x -> TConst_float32 x)
  | 12 -> u_int64 st |> (fun x -> TConst_float (Nums.bits_to_ieee64 x))
  | 13 -> u_char st |> (fun x -> TConst_char x)
  | 14 -> u_bytes st |> (fun x -> TConst_string x)
  | 15 -> u_bytes st |> (fun x -> TConst_bigint x)
  | 16 -> u_bytes st |> (fun x -> TConst_bignum x)
  | 17 -> u_void st |> (fun () -> TConst_unit)
  | 18 -> u_void st |> (fun () -> TConst_default)
  | 19 -> u_bytes st |> (fun x -> TConst_decimal x)
  | _ -> pickle_failwith "u_const" 


and pdtree x st = 
  match x with 
  | TDSwitch (a,b,c,d) -> p_byte 0 st; p_tup4 p_expr (p_list pdtree_case) (p_option pdtree) pdrange (a,b,c,d) st
  | TDSuccess (a,b)    -> p_byte 1 st; p_tup2 pexprs p_int (a,b) st
  | TDBind (a,b)       -> p_byte 2 st; p_tup2 pbind pdtree (a,b) st
and udtree st = 
  let tag = u_byte st in match tag with
  | 0 -> u_tup4 u_expr (u_list udtree_case) (u_option udtree) udrange st |> (fun (e,cases,dflt,m) -> TDSwitch (e,cases,dflt,m) ) 
  | 1 -> u_tup2 uexprs u_int st |> (fun (es,n) -> TDSuccess (es,n) )
  | 2 -> u_tup2 ubind udtree st |> (fun (b,t) -> TDBind(b,t) )
  | _ -> pickle_failwith "udtree" 

and pdtree_case (TCase(a,b)) st = p_tup2 pdtree_discrim pdtree (a,b) st
and udtree_case st = let a,b = u_tup2 udtree_discrim udtree st in (TCase(a,b)) 

and pdtree_discrim x st = 
  match x with 
  | TTest_unionconstr (ucref,tinst) -> p_byte 1 st; p_tup2 p_ucref ptyps (ucref,tinst) st
  | TTest_const c                   -> p_byte 2 st; p_const c st
  | TTest_isnull                    -> p_byte 3 st; p_void st
  | TTest_isinst (srcty,tgty)       -> p_byte 4 st; p_typ srcty st; p_typ tgty st
  | TTest_query _                   -> pickle_failwith "TTest_query: only used during pattern match compilation"
  | TTest_array_length (n,ty)       -> p_byte 6 st; p_tup2 p_int p_typ (n,ty) st
and udtree_discrim st = 
  let tag = u_byte st in match tag with
  | 1 -> u_tup2 u_ucref utyps st |> (fun (a,b) -> TTest_unionconstr (a,b) ) 
  | 2 -> u_const st             |> (fun c -> TTest_const c ) 
  | 3 -> u_void st              |> (fun () -> TTest_isnull ) 
  | 4 -> u_tup2 u_typ u_typ st    |> (fun (srcty,tgty) -> TTest_isinst (srcty,tgty) )
  | 6 -> u_tup2 u_int u_typ st    |> (fun (n,ty)       -> TTest_array_length (n,ty) )
  | _ -> pickle_failwith "udtree_discrim" 

and ptarget (TTarget(a,b)) st = p_tup2 pval_specs p_expr (a,b) st
and utarget st = let a,b = u_tup2 uval_specs u_expr st in (TTarget(a,b)) 

and pbind (TBind(a,b)) st = p_tup2 pval_spec pval_repr (a,b) st
and ubind st = let a = uval_spec st in let b = uval_repr st in TBind(a,b)

and pval_repr x st = p_expr x st
and uval_repr st = u_expr st

and plval_op_kind x st =
  p_int (match x with LGetAddr -> 0 | LByrefGet -> 1 | LSet -> 2 | LByrefSet -> 3) st
and ulval_op_kind st =
  match (u_int st) with 0 -> LGetAddr | 1 -> LByrefGet | 2 -> LSet | 3 -> LByrefSet | _ -> pickle_failwith "uval_op_kind"

and p_recdInfo x st = 
    match x with 
    | RecdExprIsObjInit -> p_byte 0 st
    | RecdExpr -> p_byte 1 st

and u_recdInfo st = 
    let tag = u_byte st in 
    match tag with 
    | 0 -> RecdExprIsObjInit 
    | 1 -> RecdExpr 
    | _ -> pickle_failwith "u_recdInfo" 
  
and pop x st = 
  match x with 
  | TOp_uconstr c                  -> p_byte 0 st; p_ucref c st
  | TOp_exnconstr c               -> p_byte 1 st; p_tcref "op"  c st
  | TOp_tuple                     -> p_byte 2 st
  | TOp_recd (a,b)                -> p_byte 3 st; p_tup2 p_recdInfo (p_tcref "recd op") (a,b) st
  | TOp_field_set (a)        -> p_byte 4 st; prfref a st
  | TOp_field_get (a)        -> p_byte 5 st; prfref a st
  | TOp_constr_tag_get (a)        -> p_byte 6 st; p_tcref "cnstr op" a st
  | TOp_constr_field_get (a,b)    -> p_byte 7 st; p_tup2 p_ucref p_int (a,b) st
  | TOp_constr_field_set (a,b)    -> p_byte 8 st; p_tup2 p_ucref p_int (a,b) st
  | TOp_exnconstr_field_get (a,b) -> p_byte 9 st; p_tup2 (p_tcref "exn op") p_int (a,b) st
  | TOp_exnconstr_field_set (a,b) -> p_byte 10 st; p_tup2 (p_tcref "exn op")  p_int (a,b) st
  | TOp_tuple_field_get (a)       -> p_byte 11 st; p_int a st
  | TOp_asm (a,b)                 -> p_byte 12 st; p_tup2 (p_list pilinstr) ptyps (a,b) st
  | TOp_get_ref_lval              -> p_byte 13 st
  | TOp_coerce                    -> p_byte 15 st
  | TOp_trait_call (b)     -> p_byte 16 st; ptrait b st
  | TOp_lval_op (a,b)             -> p_byte 17 st; p_tup2 plval_op_kind (p_vref "lval") (a,b) st
  | TOp_ilcall (a,b,c,d)          -> p_byte 18 st; p_tup4 (ptup9 p_bool p_bool p_bool p_bool pvrefFlags p_bool p_bool (p_option (p_tup2 p_typ p_typ)) p_ilmref) ptyps ptyps ptyps (a,b,c,d) st
  | TOp_array                     -> p_byte 19 st
  | TOp_while                     -> p_byte 20 st
  | TOp_for dir                   -> p_byte 21 st; p_bool dir st
  | TOp_bytes bytes               -> p_byte 22 st; p_bytes bytes st
  | TOp_try_catch                 -> p_byte 23 st
  | TOp_try_finally               -> p_byte 24 st
  | TOp_field_get_addr (a)   -> p_byte 25 st; prfref a st

and uop st = 
  let tag = u_byte st in match tag with
  | 0 -> let a = u_ucref st in 
         TOp_uconstr (a) 
  | 1 -> let a = u_tcref st in 
         TOp_exnconstr (a) 
  | 2 -> TOp_tuple 
  | 3 -> let a = u_recdInfo st in 
         let b = u_tcref st in 
         TOp_recd (a,b) 
  | 4 -> let a = urfref st in 
          TOp_field_set (a) 
  | 5 -> let a = urfref st in 
          TOp_field_get (a) 
  | 6 -> let a = u_tcref st in 
          TOp_constr_tag_get (a) 
  | 7 -> let a = u_ucref st in 
          let b = u_int st in 
          TOp_constr_field_get (a,b) 
  | 8 -> let a = u_ucref st in 
          let b = u_int st in 
          TOp_constr_field_set (a,b) 
  | 9 -> let a = u_tcref st in 
          let b = u_int st in 
          TOp_exnconstr_field_get (a,b) 
  | 10 -> let a = u_tcref st in 
          let b = u_int st in 
          TOp_exnconstr_field_set (a,b) 
  | 11 -> let a = u_int st in 
          TOp_tuple_field_get (a) 
  | 12 -> let a = (u_list uilinstr) st in 
          let b = utyps st in 
          TOp_asm (a,b) 
  | 13 -> TOp_get_ref_lval 
  | 15 -> TOp_coerce
  | 16 -> let a = utrait st in 
          TOp_trait_call a
  | 17 -> let a = ulval_op_kind st in 
          let b = u_vref st in 
          TOp_lval_op (a,b) 
  | 18 -> let (a1,a2,a3,a4,a5,a6,a7,a8,a9) = (utup9 u_bool u_bool u_bool u_bool uvrefFlags u_bool u_bool (u_option (u_tup2 u_typ u_typ)) u_ilmref) st in
          let b = utyps st in 
          let c = utyps st in 
          let d = utyps st in 
          TOp_ilcall ((a1,a2,a3,a4,a5,a6,a7,a8,a9),b,c,d) 
  | 19 -> TOp_array
  | 20 -> TOp_while
  | 21 -> TOp_for (u_bool st)
  | 22 -> TOp_bytes (u_bytes st)
  | 23 -> TOp_try_catch
  | 24 -> TOp_try_finally
  | 25 -> let a = urfref st in 
          TOp_field_get_addr (a) 
  | _ -> pickle_failwith "uop" 

and p_expr expr st = 
(* try *)
  match expr with 
  | TExpr_link e -> p_expr !e st
  | TExpr_const (x,m,ty)         -> p_byte 0 st; p_tup3 p_const pdrange p_typ (x,m,ty) st
  | TExpr_val (a,b,m)                  -> p_byte 1 st; p_tup3 (p_vref "val") pvrefFlags pdrange (a,b,m) st
  | TExpr_op(a,b,c,d)                -> p_byte 2 st; p_tup4 pop  ptyps pexprs pdrange (a,b,c,d) st
  | TExpr_seq (a,b,c,d)              -> p_byte 6 st; p_tup4 p_expr p_expr p_int pdrange (a,b,(match c with NormalSeq -> 0 | ThenDoSeq -> 1),d) st
  | TExpr_lambda (a,b0,b1,c,d,e,_)       -> p_byte 9 st; ptup6 p_int (p_option pval_spec) pval_specs p_expr pdrange p_typ (a,b0,b1,c,d,e) st
  | TExpr_tlambda (a,b,c,d,e,_)      -> p_byte 10 st; p_tup5 p_int ptypar_specs p_expr pdrange p_typ (a,b,c,d,e) st
  | TExpr_app (a1,a2,b,c,d)              -> p_byte 11 st; p_tup5 p_expr p_typ ptyps pexprs pdrange (a1,a2,b,c,d) st
  | TExpr_letrec (a,b,c,_)           -> p_byte 12 st; p_tup3 pbinds p_expr pdrange (a,b,c) st
  | TExpr_let (a,b,c,_)              -> p_byte 13 st; p_tup3 pbind p_expr pdrange (a,b,c) st
  | TExpr_match (a,b,c,d,e,_)        -> p_byte 14 st; p_tup5 pdrange pdtree ptargets pdrange p_typ (a,b,c,d,e) st
  | TExpr_obj(a,b,c,d,e,f,g,_)            -> p_byte 21 st; ptup7 p_int p_typ (p_option pval_spec) p_expr pmethods pintfs pdrange (a,b,c,d,e,f,g) st
  | TExpr_static_optimization(a,b,c,d)    -> p_byte 22 st; p_tup4 pconstraints p_expr p_expr pdrange (a,b,c,d) st
  | TExpr_tchoose (a,b,c)                 -> p_byte 25 st; p_tup3 ptypar_specs p_expr pdrange (a,b,c) st
  | TExpr_quote(raw,ast,m,ty)             -> p_byte 26 st; p_tup4 p_bool p_expr pdrange p_typ (raw,ast,m,ty) st
  | TExpr_hole(m,ty)                      -> p_byte 27 st; p_tup2 pdrange p_typ (m,ty) st

(*
with Nope -> 
   dprintf3 "\nloc: %a\nexpr: %s\n\n" output_range (Tastops.range_of_expr expr) (Layout.showL (Tastops.exprL expr));
   flush stdout;
   raise Nope
*)
and u_expr st = 
  let tag = u_byte st in match tag with
  | 0 -> let a = u_const st in 
         let b = udrange st in 
         let c = u_typ st in 
         TExpr_const (a,b,c) 
  | 1 -> let a = u_vref st in
         let b = uvrefFlags st in 
         let c = udrange st in 
         TExpr_val (a,b,c) 
  | 2 -> let a = uop st in 
         let b = utyps st in 
         let c = uexprs st in 
         let d = udrange st in 
         TExpr_op (a,b,c,d)
  | 6 -> let a = u_expr st in 
         let b = u_expr st in 
         let c = u_int st in 
         let d = udrange  st in 
         TExpr_seq (a,b,(match c with 0 -> NormalSeq | 1 -> ThenDoSeq | _ -> pickle_failwith "specialSeqFlag"),d) 
  | 9 -> let a = u_int st in 
         let b0 = u_option uval_spec st in 
         let b1 = uval_specs st in 
         let c = u_expr st in 
         let d = udrange st in 
         let e = u_typ st in 
         TExpr_lambda (a,b0,b1,c,d,e,new_cache()) 
  | 10 -> let a = u_int st in 
          let b = utypar_specs st in 
          let c = u_expr st in 
          let d = udrange st in 
          let e = u_typ st in 
          TExpr_tlambda (a,b,c,d,e,new_cache()) 
  | 11 -> let a1 = u_expr st in 
          let a2 = u_typ st in 
          let b = utyps st in 
          let c = uexprs st in 
          let d = udrange st in 
          TExpr_app (a1,a2,b,c,d) 
  | 12 -> let a = ubinds st in 
          let b = u_expr st in 
          let c = udrange st in 
          TExpr_letrec (a,b,c,new_cache()) 
  | 13 -> let a = ubind st in 
          let b = u_expr st in 
          let c = udrange st in 
          TExpr_let (a,b,c,new_cache()) 
  | 14 -> let a = udrange st in 
          let b = udtree st in 
          let c = utargets st in 
          let d = udrange st in 
          let e = u_typ st in 
          TExpr_match (a,b,c,d,e,new_cache()) 
  | 21 -> let a = u_int st in 
          let b = u_typ st in 
          let c = (u_option uval_spec) st in 
          let d = u_expr st in 
          let e = umethods st in 
          let f = uintfs st in 
          let g = udrange st in 
          TExpr_obj (a,b,c,d,e,f,g,new_cache())
  | 22 -> let a = uconstraints st in 
          let b = u_expr st in 
          let c = u_expr st in 
          let d = udrange st in 
          TExpr_static_optimization (a,b,c,d)
  | 25 -> let a = utypar_specs st in 
          let b = u_expr st in 
          let c = udrange st in 
          TExpr_tchoose (a,b,c)
  | 26 -> let a = u_bool st in 
          let b = u_expr st in 
          let c = udrange st in 
          let d = u_typ st in 
          TExpr_quote (a,b,c,d)
  | 27 -> let a = udrange st in 
          let b = u_typ st in 
          TExpr_hole(a,b)
  | _ -> pickle_failwith "u_expr" 
and parg x st = p_tup2 p_expr p_typ x st
and uarg st = u_tup2 u_expr u_typ st
and pconstraint x st = 
  match x with
  | TTyconEqualsTycon (a,b) -> p_byte 0 st; p_tup2 p_typ p_typ (a,b) st
and uconstraint st = 
  let tag = u_byte st in match tag with
  | 0 -> u_tup2 u_typ u_typ st |> (fun (a,b) -> TTyconEqualsTycon(a,b) ) 
  | _ -> pickle_failwith "uconstraint" 

and pslotparam (TSlotParam data) st = ptup6 (p_option p_string) p_typ p_bool p_bool p_bool pattribs data st
and uslotparam st = TSlotParam(utup6 (u_option u_string) u_typ u_bool u_bool u_bool uattribs st)

and pslotsig (TSlotSig data) st = 
    if verbose then dprintf0 "pslotsig\n";
    ptup6 p_string p_typ ptypar_specs ptypar_specs (p_list pslotparam) p_typ  data st
and uslotsig st = TSlotSig(utup6 u_string u_typ utypar_specs utypar_specs (u_list uslotparam) u_typ st)

and pmethod (TMethod data) st = 
    if verbose then dprintf0 "pmethod\n";
    p_tup5 pslotsig ptypar_specs pval_specs p_expr prange data st 
and umethod st = TMethod(u_tup5 uslotsig utypar_specs uval_specs u_expr urange st)

and pmethods x st = p_list pmethod x st
and umethods st = u_list umethod st

and pintf x st = p_tup2 p_typ pmethods x st
and uintf st = u_tup2 u_typ umethods st

and pintfs x st = p_list pintf x st
and uintfs st = u_list uintf st

let _ = fill_pbinds (p_list pbind);;
let _ = fill_ubinds (u_list ubind);;

let _ = fill_ptargets (p_array ptarget);;
let _ = fill_utargets (u_array utarget);;

let _ = fill_pconstraints (p_list pconstraint);;
let _ = fill_uconstraints (u_list uconstraint);;

let _ = fill_pexprs (p_list p_expr);;
let _ = fill_uexprs (u_list u_expr);;

let _ = fill_pattribs (p_list pattrib);;
let _ = fill_uattribs (u_list uattrib);;

let _ = fill_pval_specs (p_list pval_spec);;
let _ = fill_uval_specs (u_list uval_spec);;

(*---------------------------------------------------------------------------
 * Pickle/unpickle F# interface data 
 *------------------------------------------------------------------------- *)

let pickle_modul_spec mspec st = pmodul_spec mspec st
let unpickle_modul_spec st = umodul_spec st 
    
