(* (c) Microsoft Corporation. All rights reserved  *)

(* From "Algorithms", Cormen, Leiserson & Rivest p 252. *)
(* Also http://homepages.ius.edu/jholly/c343/notes/lrotate.htm *)

(*F# 
module Microsoft.Research.AbstractIL.Internal.Zmap 
open Microsoft.Research.AbstractIL 
F#*)

type 'a order = 'a -> 'a -> int

(*F#
type map<'key,'a> = Microsoft.FSharp.Collections.Tagged.Map<'key,'a>
type ('key,'a) t = map<'key,'a>

let empty (ord : 'a order) = Microsoft.FSharp.Collections.Tagged.Map<_,_,_>.Empty(ComparisonIdentity.Custom(ord))
let add k v (m:map<_,_>) = m.Add(k,v)
let find k (m:map<_,_>) = m.Find(k)
let tryfind k (m:map<_,_>) = m.TryFind(k)
let remove k (m:map<_,_>) = m.Remove(k)
let mem k (m:map<_,_>) = m.ContainsKey(k)
let iter f (m:map<_,_>) = m.Iterate(f)
let first f (m:map<_,_>) = m.First(fun k v -> if f k v then Some (k,v) else None)
let exists f (m:map<_,_>) = m.Exists(f)
let forall f (m:map<_,_>) = m.ForAll(f)
let map f (m:map<_,_>) = m.MapRange(f)
let mapi f (m:map<_,_>) = m.Map(f)
let fold f (m:map<_,_>) x = m.Fold f x
let to_list (m:map<_,_>) = m.ToList()
let fold_section lo hi f (m:map<_,_>) x = m.FoldSection lo hi f x

let is_empty (m:map<_,_>) = m.IsEmpty

let fmap f z (m:map<_,_>) =
  let m,z = m.FoldAndMap (fun k v z -> let z,v' = f z k v in v',z) z in
  z,m

let choose f  (m:map<_,_>) = m.First(f)
  
let chooseL f  (m:map<_,_>) =
  m.Fold (fun k v s -> match f k v with None -> s | Some x -> x::s) []
    
let of_list m xs = List.fold_left (fun m (k,v) -> add k v m) m xs

let keys   m = chooseL (fun k v -> Some k) m 
let values m = chooseL (fun k v -> Some v) m


F#*)


(*IF-OCAML*)
type ('key,'a) rep = E | N of 'key * 'a * ('key, 'a) rep *  ('key, 'a) rep * int
type ('key,'a) map = MAP of 'key order * ('key,'a) rep
type ('key,'a) t = ('key,'a) map 

let empty ord = MAP(ord,E)
let is_empty = function MAP(ord,E) -> true | MAP(ord,N _) -> false

let height  = function
  | E -> 0
  | N (_,_,_,_,h) -> h

let mk l k v r = 
  let hl = height l in 
  let hr = height r in 
  let m = if hl < hr then hr else hl in 
  N(k,v,l,r,m+1)

let rebalance t1 k v t2 =
  let t1h = height t1 in 
  if  height t2 > t1h + 2 then (* right is heavier than left *)
    match t2 with 
      N(t2k,t2v,t2l,t2r,t2h) -> 
        (* one of the nodes must have height > height t1 + 1 *)
        if height t2l > t1h + 1 then  (* balance left: combination *)
          match t2l with 
          | N(t2lk,t2lv,t2ll,t2lr,t2lh) ->
              mk (mk t1 k v t2ll) t2lk t2lv (mk t2lr t2k t2v t2r) 
          | _ -> failwith "rebalance"
        else (* rotate left *)
          mk (mk t1 k v t2l) t2k t2v t2r
    | _ -> failwith "rebalance"
  else
    let t2h = height t2 in 
    if  t1h > t2h + 2 then (* left is heavier than right *)
      match t1 with 
        N(t1k,t1v,t1l,t1r,t1h) -> 
        (* one of the nodes must have height > height t2 + 1 *)
          if height t1r > t2h + 1 then 
          (* balance right: combination *)
            match t1r with 
            | N(t1rk,t1rv,t1rl,t1rr,t1rh) ->
                mk (mk t1l t1k t1v t1rl) t1rk t1rv (mk t1rr k v t2)
            | _ -> failwith "rebalance"
          else
            mk t1l t1k t1v (mk t1r k v t2)
      | _ -> failwith "rebalance"
    else mk t1 k v t2

let add k v (MAP (compare,tree)) =
  let rec add k v = function 
      E -> N (k,v,E,E,1)
    | N (k2,v2,l,r,h) -> 
        let c = compare k k2 in 
        if c < 0 then rebalance (add k v l) k2 v2 r
        else if c = 0 then N(k,v,l,r,h)
        else rebalance l k2 v2 (add k v r)
  in
  MAP(compare,add k v tree)

let tryfind k (MAP (compare,tree)) =
  let rec get k = function
      E -> None
    | N(k2,v2,l,r,_) -> 
        let c = compare k k2 in 
        if c < 0 then get k l
        else if c = 0 then Some v2
        else get k r
  in
  get k tree

let find k m = match tryfind k m with Some x -> x | None -> raise Not_found 

let rec splice_out_succ = function
  | E -> failwith "internal error: map.splice_out_succ_or_pred"
  | N (k2,v2,l,r,_) ->
      match l with 
      | E -> k2,v2,r
      | N _ -> let k3,v3,l' = splice_out_succ l in k3,v3,mk l' k2 v2 r

let remove k (MAP (compare,tree)) =
  let rec remove k = function 
    | E -> E
    | N (k2,v2,l,r,_) -> 
        let c = compare k k2 in 
        if c < 0 then rebalance (remove k l) k2 v2 r
        else if c = 0 then 
          match l,r with 
          | E,_ -> r
          | _,E -> l
          | _, N(rk,rv,rl,rr,_) -> 
              let sk,sv,r' = splice_out_succ r in 
              mk l sk sv r'
        else rebalance l k2 v2 (remove k r)
  in
  MAP(compare,remove k tree)

let mem k (MAP (compare,tree)) =
  let rec mem k = function
      E -> false
    | N(k2,v2,l,r,_) -> 
        let c = compare k k2 in 
        if c < 0 then mem k l
        else if c = 0 then true
        else mem k r
  in
  mem k tree

let iter f (MAP(compare,tree)) =
  let rec iter f = function
    | E -> ()
    | N(k2,v2,l,r,_) -> iter f l; (f k2 v2 : unit); iter f r
  in
  iter f tree

let map f (MAP (compare,tree)) =
  let rec map f = function
    | E -> E
    | N(k2,v2,l,r,h) -> N(k2,f v2, map f l, map f r,h)
  in
  MAP(compare,map f tree)

let mapi f (MAP (compare,tree)) =
  let rec mapi f = function
    | E -> E
    | N(k2,v2,l,r,h) -> N(k2,f k2 v2, mapi f l, mapi f r,h)
  in
  MAP(compare,mapi f tree)

let fold f (MAP (compare,tree)) x =
  let rec fold f m x = 
    match m with 
    | E -> x
    | N(k2,v2,l,r,h) -> fold f r (f k2 v2 (fold f l x))
  in
  fold f tree x

let fold_section lo hi f (MAP (compare,tree)) x =
  let rec fold_from_to f m x =
    (* assert: compare lo hi <= 0 *)
    match m with 
    | E -> x
    | N(k,v,l,r,h) ->
        let clo_k = compare lo k in
        let ck_hi = compare k hi in            
        let x = if clo_k < 0                then fold_from_to f l x else x in
        let x = if clo_k <= 0 && ck_hi <= 0 then f k v x            else x in
        let x = if ck_hi < 0                then fold_from_to f r x else x in
        x
  in
  if compare lo hi = 1 then x else fold_from_to f tree x

let fmap f z (MAP (compare,tree)) =
  let rec fmap z m = 
    match m with 
    | E -> z,E
    | N(k2,v2,l,r,h) -> let z,l = fmap z l in
                        let z,v2 = f z k2 v2 in
                        let z,r = fmap z r in
                        z,N(k2,v2,l,r,h)
  in
  let z,tree = fmap z tree in
  z,MAP (compare,tree)

let mapni f m =
  let step n k v = n+1,f n k v in
  let n,m = fmap step 0 m in
  m

let first pred (MAP (compare,tree)) =
  let rec first m =
    match m with 
    | E -> None
    | N(k2,v2,l,r,h) ->
        match first l with
          Some x -> Some x
        | None   -> if pred k2 v2 then
                      Some (k2,v2)
                    else
                      first r
  in
  first tree

let exists pred (MAP (compare,tree)) =
  let rec exists m =
    match m with 
    | E -> false
    | N(k2,v2,l,r,h) -> exists l || pred k2 v2 || exists r
  in
  exists tree

let (|>) x f = f x
let forall pred m = exists (fun k v -> not (pred k v)) m |> not

let choose select (MAP (compare,tree)) =
  let rec choose m =
    match m with 
    | E -> None
    | N(k2,v2,l,r,h) ->
        match choose l with
          Some x -> Some x
        | None   -> match select k2 v2 with
                      Some x -> Some x
                    | None   -> choose r
  in
  choose tree

let chooseL select (MAP (compare,tree)) =
  let rec chooseL acc m =
    match m with 
    | E -> acc
    | N(k2,v2,l,r,h) -> let acc = chooseL acc r in
                        let acc = match select k2 v2 with
                                    Some x -> x::acc
                                  | None   -> acc in
                        let acc = chooseL acc l in
                        acc
  in
  chooseL [] tree

let to_list m = fold (fun k v xs -> (k,v)::xs) m []
let of_list m xs = List.fold_left (fun m (k,v) -> add k v m) m xs

let keys   m = chooseL (fun k v -> Some k) m 
let values m = chooseL (fun k v -> Some v) m
(*ENDIF-OCAML*)

let mem_of m k = mem k m
