// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 

#light

open System
open System.Threading

/// A naive imperative implementation of factorization
let factorizeImperative n = 
    let primefactor1 = ref 1
    let primefactor2 = ref n
    let i = ref 2 
    let fin = ref false
    while (!i < n && not !fin) do
        if (n % !i = 0) then 
            primefactor1 := !i;
            primefactor2 := n / !i;
            fin := true;
        i := !i + 1;

    if (!primefactor1 = 1) then None
    else Some (!primefactor1, !primefactor2)

let factorizeImperative2 n = 
    let mutable primefactor1 = 1
    let mutable primefactor2 = n
    let mutable i = 2 
    let mutable fin = false
    while (i < n && not fin) do
        if (n % i = 0) then 
            primefactor1 <- i;
            primefactor2 <- n / i;
            fin <- true;
        i <- i + 1;

    if (primefactor1 = 1) then None
    else Some (primefactor1, primefactor2)

/// The same code in "functional"/"recursive" style
let factorizeRecursive n = 
    let rec find i = 
        if i >= n then None
        elif (n % i = 0) then Some(i,n / i)
        else find (i+1)
    find 2


    
// The sample illustrates the two ways to specify what happens 
// when the computation terminates, either by explicitly calling EndInvoke on 
// the IAsyncResult operation  that acts as a handle to the asynchronous computation, 
// or (often better) by giving an AsyncCallback item to the BeginInvoke operation - the
// AsyncCallback will get invoked when the operation completes.


let n = 1000589023
  
// This loop schedules 200 an asynchronous work items in 
// the .NET thread pool,  one at a time. We wait for their execution to complete one
// at a time.  
let fd = new Converter<_,_>(factorizeImperative)
for i = 1 to 200 do  
    let ar = fd.BeginInvoke(n,null, null)
    ar.AsyncWaitHandle.WaitOne(10000, false) |> ignore 
    if ar.IsCompleted then
        match fd.EndInvoke(ar) with 
        | Some(factor1, factor2) -> 
            lock stdout (fun () -> printfn "factorizeNumber1 : Factors of %d : %d, %d" n factor1 factor2)
        | None -> 
            lock stdout (fun () -> printfn "No factors found for %d" n)
    else 
        lock stdout (fun () -> printfn "Timed out!")

// This loop schedules 200 an asynchronous work items in 
// the .NET thread pool,  all at once. We wait for their execution to complete one
// at a time.  
let ars = [ for i in 1 .. 200  
            -> let fd = (new Converter<_,_>(factorizeImperative))in 
               (fd, fd.BeginInvoke(n,null, null)) ]

// Collect all the results
ars |> List.iter (fun (fd,ar) -> 
    ar.AsyncWaitHandle.WaitOne(10000, false) |> ignore 
    if ar.IsCompleted then
        match fd.EndInvoke(ar) with 
        | Some(factor1, factor2) -> 
            lock stdout (fun () -> printfn "factorizeNumber2 : Factors of %d : %d, %d" n factor1 factor2)
        | None -> 
            lock stdout (fun () -> printfn "No factors found for %d" n)
    else 
        lock stdout (fun () -> printfn "Timed out!"))
        

// Here the completion operation is performed as in a continuation passed to BeginInvoke.
// The continuation is invoked on a .NET thread pool worker thread, so should not block.
for i = 1 to 200 do  
    let fd = new Converter<_,_>(factorizeImperative)
    fd.BeginInvoke(n,(fun ar -> 
                        match fd.EndInvoke(ar) with 
                        | Some(factor1, factor2) -> lock stdout (fun () -> printfn "AsyncCallback: Factors of %d : %d, %d" n factor1 factor2)
                        | None -> lock stdout (fun () -> printfn "No factors found for %d" n)),
                   null) |> ignore

// This routine shows that it is possible to abstract the Converter/BeginInvoke/EndInvoke pattern

// asyncApp: : (unit -> 'a) -> ('a -> unit) -> unit 
let asyncApp comp cont  =
    let fd = new Converter<_,_>(comp)
    ignore (fd.BeginInvoke((),(fun ar -> cont (fd.EndInvoke(ar))), null))


// This schedules 100 asynchronous computations and prints the results in a callback.
for i = 1 to 100 do 
    asyncApp (fun () -> factorizeImperative n)
        (fun res -> 
            match res with 
            | Some(factor1, factor2) -> lock stdout (fun () -> printfn "asyncApp: Factors of %d : %d, %d" n factor1 factor2)
            | None -> lock stdout (fun () -> printfn "No factors found for %d" n))


for i = 1 to 10 do 
    Thread.Sleep(500);
    lock stdout (fun () -> printfn "\nsleeping to allow any more async-callback workers to finish!")

lock stdout (fun () -> printfn "done!")
